<?php

/**
 * A loader class for the renderers.
 *
 * @package    linea21.externals
 * @subpackage PHP_DEBUG
 * @author     Vernet Loïc <qrf_coil[at]yahoo.fr>, modified by Linea21 <info@linea21.com>
 * @since V2.0.0 - 10 Apr 2006
 *
 * @version    CVS: $Id: DebugLine.php,v 1.1 2008/05/02 14:26:37 c0il Exp $
 */

class PHP_DebugLine
{

  /**
   * PHP_DEBUGLINE Types
   *
   * - TYPE_ANY          : All available types (for search mode)
   * - TYPE_STD          : Standart debug
   * - TYPE_QUERY        : Query debug
   * - TYPE_REL          : Database related debug
   * - TYPE_ENV          : Environment debug ($GLOBALS...)
   * - TYPE_APPERROR     : Custom application error
   * - TYPE_CREDITS      : Credits information
   * - TYPE_SEARCH       : Search mode in debug
   * - TYPE_DUMP         : Dump any kind of variable
   * - TYPE_PROCESSPERF  : Performance analysys
   * - TYPE_TEMPLATES    : Included templates of the calling script
   * - TYPE_PAGEACTION   : Store main page action
   * - TYPE_SQLPARSE     : SQL Parse error
   * - TYPE_WATCH        : A variable to watch
   * - TYPE_PHPERROR     : A debug generated by the custom error handler
   *
   * @category DebugLine
   */

  const TYPE_ANY         = 0;
  const TYPE_STD         = 1;
  const TYPE_QUERY       = 2;
  const TYPE_QUERYREL    = 3;
  const TYPE_ENV         = 4;
  const TYPE_APPERROR    = 5;
  const TYPE_CREDITS     = 6;
  const TYPE_SEARCH      = 7;
  const TYPE_DUMP        = 8;
  const TYPE_PROCESSPERF = 9;
  const TYPE_TEMPLATES   = 10;
  const TYPE_PAGEACTION  = 11;
  const TYPE_SQLPARSE    = 12;
  const TYPE_WATCH       = 13;
  const TYPE_PHPERROR    = 14;
  const TYPE_DEFAULT     = self::TYPE_STD;

  /**
   * PHP_DEBUGLINE info levels
   */
  const INFO_LEVEL    = 1;
  const WARNING_LEVEL = 2;
  const ERROR_LEVEL   = 3;

  /**
   * Labels for debugline types
   */
  public static $debugLineLabels = array(
  self::TYPE_ANY         => 'ALL',
  self::TYPE_STD         => 'Standart',
  self::TYPE_QUERY       => 'Query',
  self::TYPE_QUERYREL    => 'Database related',
  self::TYPE_ENV         => 'Environment',
  self::TYPE_APPERROR    => 'Application error',
  self::TYPE_CREDITS     => 'Credits',
  self::TYPE_SEARCH      => 'Search',
  self::TYPE_DUMP        => 'Variable dump',
  self::TYPE_PROCESSPERF => 'Performance analysis',
  self::TYPE_TEMPLATES   => 'Included files',
  self::TYPE_PAGEACTION  => 'Page main action',
  self::TYPE_SQLPARSE    => 'SQL parse error',
  self::TYPE_WATCH       => 'Watch',
  self::TYPE_PHPERROR    => 'PHP error'
  );

  /**
   * Properties that stores the non formatted debug information
   *
   * @since V2.0.0 - 11 apr 2006
   * @var string
   */
  protected $info;

  /**
   * Type of the debug information
   *
   * @since V2.0.0 - 11 apr 2006
   * @see Debug_Line constants
   * @var integer
   */
  protected $type;

  /**
   * File of debug info
   *
   * @since V2.0.0 - 11 apr 2006
   * @var integer
   */
  protected $file;

  /**
   * Line of debug info
   *
   * @since V2.0.0 - 11 apr 2006
   * @var integer
   */
  protected $line;

  /**
   * Class from witch the debug was called
   *
   * @since V2.0.0 - 13 apr 2006
   * @var integer
   */
  protected $class;

  /**
   * Function from wich the debug was called
   *
   * @var integer
   * @since V2.0.0 - 11 apr 2006
   */
  protected $function;

  /**
   * Exection time for debug info
   *
   * @var float
   * @see stopTimer()
   * @since V2.0.0 - 16 apr 2006
   */
  protected $startTime;

  /**
   * Exection end time for debug info
   *
   * @see PHP_Debug::stopTimer(), setEndTime()
   * @since V2.0.0 - 16 apr 2006
   * @var float
   */
  protected $endTime;

  /**
   * PHP_DebugLine class constructor
   *
   * Here it is set :
   * - the start time of the debug info
   * - the traceback information
   *
   * @since V2.0.0 - 11 apr 2006
   * @see PHP_Debug::add()
   */
  public function __construct($info, $type = self::TYPE_DEFAULT)
  {
    $this->setStartTime();
    $this->info = $info;
    $this->type = $type;
    $this->setTraceback();
  }

  /**
   * Fills properties of debug line with backtrace informations
   *
   * @since V2.0.0 - 15 apr 2006
   */
  protected function setTraceback()
  {
    $callStack = debug_backtrace();
    $idx = 0;

    // Get max id of 'add' debug functions
    foreach($callStack as $lkey => $lvalue) {
      if (in_array($callStack[$lkey]['function'],
      PHP_Debug::$excludedBackTraceFunctions) == true
      ) {
        $idx = $lkey;
      }
    }

    $this->file     = !empty($callStack[$idx]  ['file'])
    ? $callStack[$idx]['file']       : '';
    $this->line     = !empty($callStack[$idx]  ['line'])
    ? $callStack[$idx]['line']       : '';
    $this->function = !empty($callStack[$idx+1]['function'])
    ? $callStack[$idx+1]['function'] : '';
    $this->class    = !empty($callStack[$idx+1]['class'])
    ? $callStack[$idx+1]['class']    : '';
  }

  /**
   * Getter of all properties of Debug_Line object
   *
   * @return array    Array containg all the properties of the debugline
   * @since V2.0.0 - 21 apr 2006
   */
  public function getProperties()
  {
    return array(
            'class'     => $this->class,
            'file'      => $this->file,
            'function'  => $this->function,
            'line'      => $this->line,
            'info'      => $this->info,
            'type'      => $this->type,
            'startTime' => $this->startTime,
            'endTime'   => $this->endTime
    );
  }

  /**
   * setter of endTime
   *
   * @since V2.0.0 - 19 apr 2006
   */
  public function setEndTime($endTime = '')
  {
    $this->endTime = $endTime ? $endTime : PHP_Debug::getMicroTimeNow();
  }

  /**
   * setter of startTime
   *
   * @see pear bug http://pear.php.net/bugs/10919
   *
   * @since V2.1.2 - 04 may 2006
   */
  public function setStartTime($startTime = '')
  {
    $this->startTime = $startTime ? $startTime : PHP_Debug::getMicroTimeNow();
  }

  /**
   * Debug_Line default output function
   *
   * @since V2.0.0 - 11 apr 2006
   * @see PHP_Debug::dumpVar()
   */
  public function __toString()
  {
    return '<pre>'.
    PHP_Debug::dumpVar(
    $this,
    __CLASS__,
    false,
    PHP_DEBUG_DUMP_ARR_STR
    )
    . '</pre>';
  }

  /**
   * Function that give the debug type lable
   *
   * @author COil
   * @since  V2.0.0 - 2 apr 2007
   */
  public static function getDebugLabel($type)
  {
    return self::$debugLineLabels[$type];
  }
}