<?php
/**
 * Export module - pdf
 * 
 * Contains all functions to generate PDF file
 * 
 * @package linea21\utils\export
 * @author $Author$ - linea21 <info@linea21.com>
 * @version $Id$ 
 * @access public
 * @license http://opensource.org/licenses/gpl-3.0.html
 * Define, compose and generate a SDI (Sustainable Development Indicators) PDF report
 */

/**
 * DisplayOneSdi()
 * Affiche un IDD + infos satellites
 *
 * @param int $pointer
 * @return bool true
 */
function DisplayOneSdi($pointer)
{
  global $sdi;
  global $pdf;
  global $report_settings;

  $pdf->AddPage();
  // lien interne si actif
  if ($report_settings['summary'] === true) $pdf->SetLink($pointer+1);
  DisplayGeneralsInfo($sdi[$pointer]);
  DisplayValues($sdi[$pointer]['sdii_id']);
  if ($report_settings['display_provider'] === true) DisplayProviderInfo($sdi[$pointer]);
  if ($report_settings['display_reglementation'] === true) DisplayReglementationInfo($sdi[$pointer]);
  if ($report_settings['display_evaluation'] === true) DisplayEvaluationInfo($sdi[$pointer]);

  return true;
}


function FormatSdiItem($libelle)
{
  global $pdf;

  $libelle="        ".$libelle;
  $pdf->SetFont(DOC_POLICE, SDI_ITEM_STYLE, SDI_ITEM_SIZE);
  $pdf->SetTextColorArray(getColor('SDI_ITEM_COLOR'));
  $pdf->MultiCell(0,DOC_HEIGHT,$libelle,0,'L',0);
  $pdf->SetTextColorArray(getColor('DOC_COLOR'));
  $pdf->SetFont(DOC_POLICE,DOC_STYLE,DOC_SIZE);
}

function FormatSdiInfo($libelle)
{
  global $pdf;

  $pdf->SetFillColorArray(getColor('BCKG_INFO'));
  $pdf->SetFont(DOC_POLICE, SDI_INFO_STYLE, SDI_INFO_SIZE);
  $pdf->SetTextColorArray(getColor('SDI_INFO_COLOR'));
  $pdf->MultiCell(0,DOC_HEIGHT+2,'   '.$libelle,0,'L',1, 1, '', '', true, 0, false, true, DOC_HEIGHT+2, 'M');
//   $pdf->SetLineStyle(array('color'=> array(100,100,100)));
//   $pdf->Line($pdf->GetX(), $pdf->GetY(), $pdf->GetX()+180, $pdf->GetY());
  $pdf->SetTextColorArray(getColor('DOC_COLOR'));
  $pdf->SetFont(DOC_POLICE,DOC_STYLE,DOC_SIZE);
  $pdf->Ln(DOC_INTERVAL);
}

// Saut de paragraphe
function ParagraphBreak()
{
  global $pdf;

  $pdf->SetTextColorArray(getColor('DOC_COLOR'));
  $pdf->SetFont(DOC_POLICE,DOC_STYLE,DOC_SIZE);
  $pdf->Ln(PARAGRAPH_BREAK);
}

/**
 * DisplayGeneralsInfo()
 * Affiche les infos générales d'un IDD
 *
 * @param int $current_record
 * @return bool true
 */
function DisplayGeneralsInfo($current_record)
{
  global $pdf;
  global $lang;
  global $report_settings;

  // titre d'un indicateur
  $out_title = formatText($current_record['sdii_name']);
  $pdf->SetTextColorArray(getColor('SDI_TITLE_COLOR'));
  $pdf->SetFont(DOC_POLICE, SDI_TITLE_STYLE, SDI_TITLE_SIZE);
  $pdf->Bookmark(mb_ucfirst(strtolower($out_title)));
  $pdf->Write(DOC_HEIGHT, $out_title);
  // si non publié
  if ($current_record['sdii_statut'] == 'D') {
    $pdf->SetTextColorArray(getColor('DOC_ADVISE_COLOR'));
    $pdf->SetFont(DOC_POLICE, '', SDI_TITLE_SIZE);
    $out_status = ' / ' . strtolower($lang['statut']['draftpdf']);
    $pdf->Write(DOC_HEIGHT, $out_status);
    $pdf->SetFont(DOC_POLICE, SDI_TITLE_STYLE, SDI_TITLE_SIZE);
    $pdf->SetTextColorArray(getColor('DOC_COLOR'));
  }
  // si non attaché au tableau de bord
  if ($current_record['sdii_to_dashboard'] == 'N') {
  	$pdf->SetTextColorArray(getColor('DOC_ADVISE_COLOR'));
  	$pdf->SetFont(DOC_POLICE, '', SDI_TITLE_SIZE);
  	$out_attached_to_dashboard = ' (' . strtolower($lang['sdi']['not_attached_to_dashboard'].')');
  	$pdf->Write(DOC_HEIGHT, $out_attached_to_dashboard);
  	$pdf->SetFont(DOC_POLICE, SDI_TITLE_STYLE, SDI_TITLE_SIZE);
  	$pdf->SetTextColorArray(getColor('DOC_COLOR'));
  }

  $pdf->Ln(DOC_HEIGHT);
  // thème d'appartenance
  $out_theme = formatText('» ' . mb_ucfirst($lang['sdi']['theme']) . ' : ' . $current_record['theme_name'] . ' / ' . mb_ucfirst($lang['sdi']['sdtheme']) . ' : ' . $current_record['sdtheme_name']) . PHP_EOL;
  $pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);
  $pdf->SetTextColorArray(getColor('SDI_TITLE_COLOR'));
  $pdf->Write(DOC_HEIGHT, $out_theme);
  $pdf->SetY($pdf->GetY() + DOC_INTERVAL);
  $pdf->SetTextColorArray(getColor('DOC_COLOR'));
  // description
  $out_desc = formatText(strip_tags($current_record['sdii_description'])) . PHP_EOL;
	$converted = new html2text($current_record['sdii_description']);
  $out_desc = formatText($converted->get_text()) . PHP_EOL;
  $pdf->Bookmark(mb_ucfirst($lang['sdi']['description']), 1, -1);
  $pdf->MultiCell(0, DOC_HEIGHT, $out_desc, 0, 'L');
  $pdf->SetY($pdf->GetY() + DOC_INTERVAL);
  // objectif
//   $out_goal = formatText(empty_nc(strip_tags($current_record['sdii_goal']))) . PHP_EOL;
  $converted = new html2text($current_record['sdii_goal']);
  $out_goal = formatText(empty_nc($converted->get_text())) . PHP_EOL;
  $out_goal_item = formatText(mb_ucfirst($lang['sdi']['goal']));
  $pdf->Bookmark($out_goal_item, 1, -1);
  FormatSdiItem($out_goal_item);
  $pdf->MultiCell(0, DOC_HEIGHT, $out_goal, 0, 'L');
  $pdf->SetY($pdf->GetY() + DOC_INTERVAL);
  // conseil d'amélioration
//   $out_consulting = formatText(empty_nc(strip_tags($current_record['sdii_consulting']))) . PHP_EOL;
  $converted = new html2text($current_record['sdii_consulting']);
  $out_consulting = formatText(empty_nc($converted->get_text())) . PHP_EOL;
  $out_consulting_item = formatText(mb_ucfirst($lang['sdi']['consulting']));
  $pdf->Bookmark($out_consulting_item, 1, -1);
  FormatSdiItem($out_consulting_item);
  $pdf->MultiCell(0, DOC_HEIGHT, $out_consulting, 0, 'L');
  $pdf->SetY($pdf->GetY() + DOC_INTERVAL);


  // informations de mesure
  $out_mesures_item = formatText(ucfirst($lang['sdi']['info_mesure']));
  $out_mesures_behavior = ($current_record['sdii_threshold_relative'] == 'Y' ? $lang['sdi']['threshold_relative_Y']: $lang['sdi']['threshold_relative_N']);
  $out_mesures_unit = formatText(mb_ucfirst($lang['sdi']['unit'])) . ' : ' . $current_record['sdii_unit'] . PHP_EOL;
  $pdf->Bookmark($out_mesures_item, 1, -1);
  FormatSdiItem($out_mesures_item);
  // unitée de mesure
  $pdf->SetFont(DOC_POLICE, EM_STYLE, MIN_SIZE);
  $pdf->Write(DOC_HEIGHT, $out_mesures_unit . PHP_EOL);
  // comportement
  $pdf->Write(DOC_HEIGHT, mb_ucfirst($out_mesures_behavior) . PHP_EOL);
  $pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);
  // tableau de valeurs
  DrawMesuresArray($current_record);
  $pdf->SetFont(DOC_POLICE, EM_STYLE, MIN_SIZE);
  $pdf->SetY($pdf->GetY() + DOC_INTERVAL);
}

/**
 * DrawMesuresArray()
 * Dessine le tableau des valeurs IDD
 *
 * @param  $current_record
 * @return bool true
 */
function DrawMesuresArray($current_record)
{

  global $pdf;
  global $lang;
  
  $a = getBooleanValues($current_record);
  
  // if indicator is boolean and mask is defined
  if($current_record['sdii_type'] == 'boolean' && $a) {
  	$display_minvalue = $a[0];
  	$display_maxvalue = $a[1];
  } else {
  	$display_minvalue = empty_nc($current_record['sdii_min_value']);
  	$display_maxvalue = empty_nc($current_record['sdii_max_value']);
  }

  $tbl = '<style>
    td {
    	text-align:center;
    }
	</style>
  <table cellspacing="0" cellpadding="5" border="1" nobr="true">
  <thead>
      <tr>
          <td>'.mb_ucfirst($lang['sdi']['min_value']).'</td>
          <td>'.mb_ucfirst($lang['sdi']['max_value']).'</td>
          <td>'.mb_ucfirst($lang['sdi']['initial_threshold_value']).'</td>
          <td>'.mb_ucfirst($lang['sdi']['frequency']).'</td>
      </tr>
  </thead>
      <tr>
          <td>'.$display_minvalue.'</td>
          <td>'.$display_maxvalue.'</td>
          <td>'.empty_nc($current_record['sdii_threshold_value']).'</td>
          <td>'.$current_record['sdii_frequency'].'</td>
      </tr>
  </table>';

  $pdf->SetFont(DOC_POLICE, DOC_STYLE, MIN_SIZE);
  $pdf->writeHTML($tbl, true, false, false, false, '');
  $pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);

  return true;
}

/**
 * DisplayProviderInfo()
 * Affiche les informations Fournisseurs d'un IDD
 *
 * @param  $current_record
 * @return bool true
 */
function DisplayProviderInfo($current_record)
{
  global $pdf;
  global $lang;

  $out_name = formatText($current_record['sdip_name']);
  if (!empty($current_record['sdip_service'])) $out_name .= ' - ' . formatText($current_record['sdip_service']);
  if (!empty($current_record['sdip_incharge'])) $out_name .= ' - ' . formatText($current_record['sdip_incharge']) . ' (' . $lang['sdi']['p_incharge'] . ')';
  $out_name .= PHP_EOL;
//   $out_adress = formatText(empty_nc(strip_tags($current_record['sdip_address'])));
  $converted = new html2text($current_record['sdip_address']);
  $out_adress = formatText(empty_nc($converted->get_text()));
  $out_phone_fax = mb_ucfirst($lang['sdi']['p_phone']) . ' : ' . formatText(empty_nc($current_record['sdip_phone']));
  $out_phone_fax .= ' - ' . mb_ucfirst($lang['sdi']['p_fax']) . ' : ' . formatText(empty_nc($current_record['sdip_fax'])) . PHP_EOL;
  $out_email_item = mb_ucfirst($lang['sdi']['p_email']) . ' : ';
  $out_email = formatText($current_record['sdip_email']);
//   $out_desc = formatText(empty_nc(strip_tags($current_record['sdip_description']))) . PHP_EOL;
  $converted = new html2text($current_record['sdip_description']);
  $out_desc = formatText(empty_nc($converted->get_text())) . PHP_EOL;
  $out_provider_item = formatText(ucfirst($lang['sdi']['p_title'])) ;

  ParagraphBreak();
  $pdf->Bookmark($out_provider_item, 1, -1);
  FormatSdiInfo($out_provider_item);

  $firstcell_width = 60;

  $pdf->SetX($pdf->GetX() + $firstcell_width);
  $pdf->MultiCell(0, DOC_HEIGHT, $out_name, 0, 'L');

  $pdf->SetX($pdf->GetX() + $firstcell_width);
  $pdf->MultiCell(0, DOC_HEIGHT, $out_adress, 0, 'L');

  $pdf->SetX($pdf->GetX() + $firstcell_width);
  $pdf->MultiCell(0, DOC_HEIGHT, $out_phone_fax, 0, 'L');

  $pdf->SetX($pdf->GetX() + $firstcell_width);
  $pdf->Cell($pdf->GetStringWidth($out_email_item), DOC_HEIGHT, $out_email_item);
  if (empty($out_email)) $pdf->Cell(0, DOC_HEIGHT, empty_nc($out_email));
  else $pdf->Cell(0, DOC_HEIGHT, $out_email, 0, 0, 'L', 0, 'mailto:' . $out_email);
  $pdf->SetY($pdf->GetY() + DOC_INTERVAL);

  return true;
}

/**
 * DisplayReglementationInfo()
 * Affiche la réglementation d'un IDD
 *
 * @param  $current_record
 * @return bool true
 */
function DisplayReglementationInfo($current_record)
{
  global $pdf;
  global $lang;

  $out_name = formatText(empty_nc($current_record['sdir_title'])) . PHP_EOL;
//    $out_body = formatText(empty_nc(strip_tags($current_record['sdir_body']))) . PHP_EOL;
  $converted = new html2text($current_record['sdir_body']);
  $out_body = formatText(empty_nc($converted->get_text())) . PHP_EOL;
  $out_uri = formatText($current_record['sdir_mask_uri']);
  $out_uri_href = $current_record['sdir_referer_uri'];

  $out_name_item = mb_ucfirst($lang['sdi']['r_title']) . ' : ';
  $out_body_item = mb_ucfirst($lang['sdi']['r_body']) . ' : ';
  $out_uri_item = mb_ucfirst($lang['sdi']['r_referer_uri']) . ' : ';

  $out_reglementation_item = formatText(ucfirst($lang['sdi']['add_step4'])) ;

  ParagraphBreak();
  $pdf->Bookmark($out_reglementation_item, 1, -1);
  FormatSdiInfo($out_reglementation_item);

  $firstcell_width = 60;
  // nom de l'organisme fournisseur
  $pdf->SetFont(DOC_POLICE, EM_STYLE, DOC_SIZE);
  $pdf->Cell($firstcell_width, DOC_HEIGHT, $out_name_item, 0, 0, 'R');
  $pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);
  $pdf->MultiCell(0, DOC_HEIGHT, $out_name, 0, 'L');
  $pdf->SetY($pdf->GetY() + DOC_INTERVAL);
  // détail reglementation
  $pdf->SetFont(DOC_POLICE, EM_STYLE, DOC_SIZE);
  $pdf->Cell($firstcell_width, DOC_HEIGHT, $out_body_item, 0, 0, 'R');
  $pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);
  $pdf->MultiCell(0, DOC_HEIGHT, $out_body, 0, 'L');
  $pdf->SetY($pdf->GetY() + DOC_INTERVAL);
  // lien internet reglementation
  $pdf->SetFont(DOC_POLICE, EM_STYLE, DOC_SIZE);
  $pdf->Cell($firstcell_width, DOC_HEIGHT, $out_uri_item, 0, 0, 'R');
  $pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);
  if (!empty($out_uri_href)) {
    if (!empty($out_uri)) $pdf->Cell(0, DOC_HEIGHT, $out_uri, 0, 0, '', 0, $out_uri_href);
    //if (!empty($out_uri)) $pdf->Cell(0, DOC_HEIGHT, ' (', 0, 0);
    //$pdf->Cell(0, DOC_HEIGHT, $out_uri, 0, 0, '', 0, $out_uri_href);
    //if (!empty($out_uri)) $pdf->Cell(0, DOC_HEIGHT, ')', 0, 0);
  } else $pdf->Write(DOC_HEIGHT, empty_nc($out_uri));
  $pdf->SetY($pdf->GetY() + DOC_INTERVAL);

  return true;
}

/**
 * DisplayEvaluationInfo()
 * Affiche le tableau de critères d'évaluation
 *
 * @param int $current_record
 * @return bool true
 */
function DisplayEvaluationInfo($current_record)
{
  global $pdf;
  global $lang;

  $out_eval_item = formatText(mb_ucfirst($lang['sdi']['e_title'])) ;

  ParagraphBreak();
  $pdf->Bookmark($out_eval_item, 1, -1);
  FormatSdiInfo($out_eval_item);
  $pdf->SetX($pdf->GetX());


  $tbl = '<style>
		    td {
		    	text-align:center;
		    }
			</style>
      <table cellspacing="0" cellpadding="5" border="1" nobr="true">
          <tr>
              <td>'.mb_ucfirst($lang['sdi']['e_scale_compare']).'</td>
              <td>'.mb_ucfirst($lang['sdi']['e_fiability']).'</td>
              <td>'.mb_ucfirst($lang['sdi']['e_accessibility']).'</td>
              <td>'.mb_ucfirst($lang['sdi']['e_lisibility']).'</td>
              <td>'.mb_ucfirst($lang['sdi']['e_relevance']).'</td>
              <td>'.mb_ucfirst($lang['sdi']['e_global_performance']).'</td>
          </tr>
        <tr>
        	<td>'.$current_record['sdie_scale_compare'].'</td>
            <td>'.$current_record['sdie_fiability'].'</td>
            <td>'.$current_record['sdie_accessibility'].'</td>
            <td>'.$current_record['sdie_lisibility'].'</td>
            <td>'.$current_record['sdie_relevance'].'</td>
            <td>'.$current_record['sdie_global_performance'].'</td>
      	</tr>
        </tbody>
      </table>';

  $pdf->SetFont(DOC_POLICE, DOC_STYLE, MIN_SIZE);
  $pdf->writeHTML($tbl, true, false, false, false, '');
  $pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);

  return true;
}


/**
 * DisplayValues()
 * Affiche les valeurs d'un IDD.
 *
 * @param int $sdi_id
 * @return bool true
 **/
function DisplayValues($sdi_id)
{
  global $pdf;
  global $lang;
  global $report_settings;
  global $scale_array;
  global $sql_object;
  global $scale_denomination_array;

  if(count($scale_array) == 0) return true;

  if ($report_settings['graphic_values'] === true && $report_settings['table_values'] === true) $mode = 'TWICE';
  elseif ($report_settings['graphic_values'] === true || $report_settings['table_values'] === true) $mode = 'ONE';
  else return false;

  $out_values_item = mb_ucfirst($lang['dashboard']['value']);
  ParagraphBreak();
  $pdf->Bookmark($out_values_item, 1, -1);
  FormatSdiInfo($out_values_item);

  $scale_no_values = array(); // recipient no values
  $scale_insuffisant_values = array(); // recipient insuffisant values graph


  for($i = 0; $i < count($scale_array); $i++) {

  	// we print chart if asked
    if ($report_settings['graphic_values'] === true) {
      if (DisplayGraphicValues($sdi_id, $scale_array[$i], GetStatus($report_settings['status']), $mode) === false) array_push($scale_insuffisant_values, $scale_array[$i]);
    }

    // we print values if asked
    if ($report_settings['table_values'] === true) {
    	if (DisplayTableValues($sdi_id, $scale_array[$i], GetStatus($report_settings['status']), $mode) === false) array_push($scale_no_values, $scale_array[$i]);
    }

  }
  DisplayNoValuesFor($scale_no_values, $lang['dashboard']['rap_no_values_for']);
  DisplayNoValuesFor($scale_insuffisant_values, $lang['dashboard']['rap_insuffisant_values'], $mode, $scale_no_values);

  return true;
}


/**
 * GetStatus()
 * Formattage du statut pour requête SQL
 *
 * @param string $current_status
 * @return string $status
 **/
function GetStatus($current_status)
{
  switch ($current_status) {
    case 'ALL':
      $status = 'SCA';
      break;
    case 'PUBLIC':
      $status = 'SCP';
      break;
  }
  return $status;
}

/**
 * DisplayNoValuesFor()
 * Affiche les échelles ne contenant pas de valeurs ou insuffisamment
 *
 * @param array $array_scale
 * @param string $message
 * @param integer $mode
 * @param integer $array
 * @return bool true
 **/
function DisplayNoValuesFor($array_scale, $message, $mode = -1, $array = -1)
{
  global $pdf;
  global $lang;
  global $scale_denomination_array;
  $content = '';
  $sep = '';
  // si mode TWICE et deuxième tableau fourni. On le soustrait au premier.
  if ($array != -1 && $mode == 'TWICE') $array_scale = array_values(array_diff($array_scale, $array));

  if (count($array_scale) == 0) return true;

  for($i = 0; $i < count($array_scale); $i++) {
    $content .= $sep . $scale_denomination_array[$array_scale[$i]];
    $sep = ', ';
  }
  if (empty($content)) return true;
  $margin = 0;
  $pdf->SetX($pdf->GetX() + $margin);
  $pdf->SetFont(DOC_POLICE, EM_STYLE, DOC_SIZE);
  $pdf->Cell(0, DOC_HEIGHT, $message, 0, 1, 'L');
  $pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);
  $margin = 10;
  $pdf->SetTextColorArray(getColor('DOC_ADVISE_COLOR'));
  $pdf->SetX($pdf->GetX() + $margin);
  $pdf->MultiCell(0, DOC_HEIGHT, $content, 0, 'L');
  $pdf->SetY($pdf->GetY() + DOC_INTERVAL);
  $pdf->SetTextColorArray(getColor('DOC_COLOR'));

  return true;
}

/**
 * DisplayGraphicValues()
 * Gènère et affiche les valeurs sous forme graphique
 *
 * @param int $sdi_id
 * @param int $current_scale
 * @param string $status
 * @param string $mode
 * @return bool true
 **/
function DisplayGraphicValues($sdi_id, $current_scale, $status, $mode)
{
	global $pdf;
	global $sql_object;
	global $scale_denomination_array;
	

	include_once(override('../dashboard/graph.php'));
	
 	$values = $sql_object->DBSelect(SQL_getAllValue($status, $current_scale, $sdi_id));
	
// 	// renversement des valeurs pour affichage chronologique
// 	if(is_array($values)) $values = @array_reverse ($values, false);

// 	$result_sdii = $sql_object->DBSelect(SQL_getInfoSdi($sdi_id));

	if (count($values) < 3) 	return false;

	generateGraphic($sdi_id, $current_scale, $status, 'auto', false, '../tmp/report');
	
	// /////////////////////////////
	// Affichage

	// définition du positionnement
	$_y = $pdf->GetY() + 5;
	$_x = 60;

	// Affichage du titre
	$out_scale_item = $scale_denomination_array[$current_scale];
	//$pdf->Bookmark($out_scale_item, 2, -1);
	FormatSdiItem($out_scale_item);
	$pdf->SetY($pdf->GetY() + DOC_INTERVAL * 2);

	// Affichage de l'image
	$pdf->Image('../tmp/report/'.$sdi_id.'_'.$current_scale.'.png', $_x, $_y, GRAPH_WIDTH * 0.30, GRAPH_HEIGHT * 0.30);

	// on place le curseur en bas du graphique
	$pdf->SetY($pdf->GetY() + (GRAPH_HEIGHT * 0.30));

	return true;
	
}

/**
 * DisplayTableValues()
 * Affiche un tableau de valeurs a une échelle donnée
 *
 * @param int $sdi_id
 * @param int $current_scale
 * @param string $status
 * @param string $mode
 * @return bool true
 **/
function DisplayTableValues($sdi_id, $current_scale, $status, $mode)
{
	
	global $pdf;
	global $lang;
	global $sql_object;
	global $scale_denomination_array;

	$values ='';

	// setting $default_threshold value
	// By default initial value
	$result_sdii = $sql_object -> DBSelect(SQL_getInfoSdi($sdi_id));
	$default_threshold = $result_sdii[0]['sdii_threshold_value'];

	$sdi_values = $sql_object->DBSelect(SQL_getAllValue($status, $current_scale, $sdi_id));
	

	if (count($sdi_values) <= 1 && $sdi_values[0] == false) return false;

	// we prepare values for displaying
	for($i = 0; $i < count($sdi_values); $i++) {

		if(!empty($sdi_values[$i]['sdiv_multivalue'])) {

			$data = unserialize($sdi_values[$i]['sdiv_multivalue']);
			$current_value = '<p style="text-align:left;">';

			foreach($data as &$val) {
				$current_value .= '';
				if(!empty($val['label'])) $current_value .= $val['label']. ' : ';
				$current_value .= $val['value']. '<span class="value-percent"> ('.fnumber_format($val['_percentage'], 2, false).' %)</span>';
				$current_value .= '<br />';
			}
			// we display the total
			$current_value .= '<strong>'. mb_ucfirst(_t('dashboard', 'multivalue_total')) . ' : ' . $data[0]['_total']. '</strong>';
			$current_value .= '</p>';
		} else {
			
			$current_value = '';
			
			// If indicator is boolean
			if($result_sdii[0]['sdii_type'] == 'boolean') {
				$a = getBooleanValues($result_sdii[0]);
				if($a) {
					$current_value .= $a[$sdi_values[$i]['sdiv_value']];
				} else {
					$current_value .= fnumber_format($sdi_values[$i]['sdiv_value'], 'auto', false);
				}
				 
			} else {
				$current_value .= fnumber_format($sdi_values[$i]['sdiv_value'], 'auto', false);
			}
			
		}
		 
		// if set, we get it from the table
		if(!is_null($sdi_values[$i]['sdiv_threshold'])) {
			$threshold = fnumber_format($sdi_values[$i]['sdiv_threshold'], 'auto', false);
		} else {
			$threshold = fnumber_format($default_threshold, 'auto', false);
		}
		 
		if ($sdi_values[$i]['sdiv_statut'] == 'D') $current_value .= ' *';
		if (!empty($sdi_values[$i]['sdiv_comment'])) {
			if($sdi_values[$i]['sdiv_comment_display']=='Y') {
				$comment_status = _t('dashboard', 'public');
			} else {
				$comment_status = _t('dashboard', 'private');
			}
			
			// $comments = formatText(strip_tags($sdi_values[$i]['sdiv_comment'])) .' ('.$comment_status.')<br />';
			$converted = new html2text($sdi_values[$i]['sdiv_comment']);
			$comments = formatText($converted->get_text())  .' ('.$comment_status.')<br />';

		} else {
			$comments = '';
		}
		
		$current_date = formatText($sdi_values[$i]['date_p']);
		$values .= '
		<tr nobr="true">
		<td class="center">'.$current_value.'</td>
		<td class="center">'.$threshold.'</td>
		<td class="center">'.$current_date.'</td>
		<td class="comment">'.$comments.'</td>
		</tr>';
	}


	$tbl = '<style>
		table {}
		td {
			border-bottom:1px solid #000;
		}
    tr.header td {
      font-weight:bold;
      text-align:center;
    }

    td.center {
    	text-align:center;
    }
    td.comment {
    }
	</style>
	<table cellspacing="0" cellpadding="5">
	<thead>
	<tr class="header">
	<td>'.mb_ucfirst($lang['dashboard']['name_pluriel']).'</td>
	<td>'.mb_ucfirst($lang['dashboard']['threshold_value']).'</td>
	<td>'.mb_ucfirst($lang['statut']['published_on']).'</td>
	<td class="comment">'.mb_ucfirst($lang['sdi']['comment']).'</td>
	</tr>
	</thead>
	'.$values.'
	</table>';

	// Affichage des données

	// Affichage du titre seulement si affichage des valeurs en standalone
	// ou si la génération des graphiques est infaisables car moins de 3 valeurs
	if ($mode != 'TWICE' || (count($sdi_values) >= 1 && count($sdi_values) < 3)) {
		$out_scale_item = $scale_denomination_array[$current_scale];
		//$pdf->Bookmark($out_scale_item, 2, -1);
		FormatSdiItem($out_scale_item);
		$pdf->SetY($pdf->GetY() + DOC_INTERVAL * 2);
	}
	
	$pdf->SetFont(DOC_POLICE, DOC_STYLE, ARRAY_DATA_SIZE );
	// Affichage du tableau
	$pdf->writeHTML($tbl, true, false, false, false, '');
	// Affichage de l'astérisque - avertissement
	$pdf->MultiCell(0, DOC_HEIGHT, '* ' . strtolower($lang['statut']['draftpdf']), 0, 'L', 0);
	$pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);
	$pdf->Ln(DOC_INTERVAL * 2);
		
	return true;
}

?>