<?php
/**
 * Export module - word
 * 
 * Contains all functions to generate Word file
 * 
 * @package linea21\utils\export
 * @author $Author$ - linea21 <info@linea21.com>
 * @version $Id$ 
 * @access public
 * @license http://opensource.org/licenses/gpl-3.0.html
 * Define, compose and generate a SDI (Sustainable Development Indicators) PDF report
 */

/**
 * DisplayOneSdi()
 * Affiche un IDD + infos satellites
 *
 * @param int $pointer
 * @return bool true
 */
function DisplayOneSdi($pointer)
{
  global $sdi;
  global $section;
  global $report_settings;
  global $pageBreak;


  DisplayGeneralsInfo($sdi[$pointer]);
  DisplayValues($sdi[$pointer]['sdii_id']);
  if ($report_settings['display_provider'] === true) DisplayProviderInfo($sdi[$pointer]);
  if ($report_settings['display_reglementation'] === true) DisplayReglementationInfo($sdi[$pointer]);
  if ($report_settings['display_evaluation'] === true) DisplayEvaluationInfo($sdi[$pointer]);

  // because $section->addPageBreak(); does not work in current version
  $section->addTextBreak(null, null, $pageBreak);
  
  return true;
}


/**
 * DisplayGeneralsInfo()
 * Affiche les infos générales d'un IDD
 *
 * @param int $current_record
 * @return bool true
 */
function DisplayGeneralsInfo($current_record)
{
  global $section;
  global $lang;
  global $report_settings;

  // titre d'un indicateur
  $out_title = formatText($current_record['sdii_name']);

  // si non publié
  if ($current_record['sdii_statut'] == 'D') {
    $out_title .= ' / ' . strtolower($lang['statut']['draftpdf']);
  }
  // si non attaché au tableau de bord
  if ($current_record['sdii_to_dashboard'] == 'N') {
  	$out_title .= ' (' . strtolower($lang['sdi']['not_attached_to_dashboard'].')');
  }
  
  $section->addTitle($out_title, 1);
  
  // thème d'appartenance
  $out_theme = formatText('» ' . mb_ucfirst($lang['sdi']['theme']) . ' : ' . $current_record['theme_name'] . ' / ' . mb_ucfirst($lang['sdi']['sdtheme']) . ' : ' . $current_record['sdtheme_name']) . PHP_EOL;
  $section->addText($out_theme, 'attachedTheme');
  
  // description
  $out_desc = formatText(do_strip($current_record['sdii_description'])) . PHP_EOL;
  $section->addText($out_desc);
  $section->addTextBreak(1);
  
  // objectif
  $out_goal = formatText(empty_nc(do_strip($current_record['sdii_goal']))) . PHP_EOL;
  $out_goal_item = formatText(mb_ucfirst($lang['sdi']['goal']));
  $section->addTitle($out_goal_item, 2);
  
  $section->addText($out_goal);
  $section->addTextBreak(1);
  
  // conseil d'amélioration
  $out_consulting = formatText(empty_nc(do_strip($current_record['sdii_consulting']))) . PHP_EOL;
  $out_consulting_item = formatText(mb_ucfirst($lang['sdi']['consulting']));
  $section->addTitle($out_consulting_item, 2);
  $section->addText($out_consulting);
  $section->addTextBreak(1);

  // informations de mesure
  $out_mesures_item = formatText(ucfirst($lang['sdi']['info_mesure']));
  $out_mesures_behavior = ($current_record['sdii_threshold_relative'] == 'Y' ? $lang['sdi']['threshold_relative_Y']: $lang['sdi']['threshold_relative_N']);
  $out_mesures_unit = formatText(mb_ucfirst($lang['sdi']['unit'])) . ' : ' . $current_record['sdii_unit'] . PHP_EOL;
  $section->addTitle($out_mesures_item, 2);

  // unitée de mesure
  $section->addText($out_mesures_unit);
  // $section->addTextBreak(1);
  // comportement
  $section->addText(mb_ucfirst($out_mesures_behavior));
  // tableau de valeurs
  DrawMesuresArray($current_record);
  $section->addTextBreak(1);
}

/**
 * DrawMesuresArray()
 * Dessine le tableau des valeurs IDD
 *
 * @param  $current_record
 * @return bool true
 */
function DrawMesuresArray($current_record)
{

  global $section;
  global $phpWord;
  global $lang;
  global $cellalign;
  global $styleTable;
  global $firstRowStyle;
  
  $a = getBooleanValues($current_record);
  
  // if indicator is boolean and mask is defined
  if($current_record['sdii_type'] == 'boolean' && $a) {
  	$display_minvalue = $a[0];
  	$display_maxvalue = $a[1];
  } else {
  	$display_minvalue = empty_nc($current_record['sdii_min_value']);
  	$display_maxvalue = empty_nc($current_record['sdii_max_value']);
  }
  
  $fStyle = array('size' => FONT_SIZE_MIN);
  $cStyle = array('align' => 'center');
  
  $phpWord->addTableStyle('units', $styleTable, $firstRowStyle);
  $table = $section->addTable('units');
  

	// row 1 - header
	$row = $table->addRow();
	$row->addCell(PAGE_WIDTH * 0.25, $cellalign)->addText(mb_ucfirst($lang['sdi']['min_value']), 'arrayHeader', $cStyle);
	$row->addCell(PAGE_WIDTH * 0.25, $cellalign)->addText(mb_ucfirst($lang['sdi']['max_value']), 'arrayHeader', $cStyle);
	$row->addCell(PAGE_WIDTH * 0.25, $cellalign)->addText(mb_ucfirst($lang['sdi']['initial_threshold_value']), 'arrayHeader', $cStyle);
	$row->addCell(PAGE_WIDTH * 0.25, $cellalign)->addText(mb_ucfirst($lang['sdi']['frequency']), 'arrayHeader', $cStyle);
	
	// row 2
	$row = $table->addRow();
	$row->addCell(PAGE_WIDTH * 0.25, $cellalign)->addText($display_minvalue, $fStyle, $cStyle);
	$row->addCell(PAGE_WIDTH * 0.25, $cellalign)->addText($display_maxvalue, $fStyle, $cStyle);
	$row->addCell(PAGE_WIDTH * 0.25, $cellalign)->addText(empty_nc($current_record['sdii_threshold_value']), $fStyle, $cStyle);
	$row->addCell(PAGE_WIDTH * 0.25, $cellalign)->addText($current_record['sdii_frequency'], $fStyle, $cStyle);

  return true;
}

/**
 * DisplayProviderInfo()
 * Affiche les informations Fournisseurs d'un IDD
 *
 * @param  $current_record
 * @return bool true
 */
function DisplayProviderInfo($current_record)
{
  global $section;
  global $lang;
  
  $lwidth = 0.29;

  $out_name = formatText($current_record['sdip_name']);
  if (!empty($current_record['sdip_service'])) $out_name .= ' - ' . formatText($current_record['sdip_service']);
  if (!empty($current_record['sdip_incharge'])) $out_name .= ' - ' . formatText($current_record['sdip_incharge']) . ' (' . $lang['sdi']['p_incharge'] . ')';
  $out_adress = formatText(empty_nc(do_strip($current_record['sdip_address'])));
  $out_phone_fax = mb_ucfirst($lang['sdi']['p_phone']) . ' : ' . formatText(empty_nc($current_record['sdip_phone']));
  $out_phone_fax .= ' - ' . mb_ucfirst($lang['sdi']['p_fax']) . ' : ' . formatText(empty_nc($current_record['sdip_fax']));
  $out_email_item = mb_ucfirst($lang['sdi']['p_email']) . ' : ';
  $out_email = formatText($current_record['sdip_email']);
  $out_desc = formatText(empty_nc(do_strip($current_record['sdip_description'])));
  $out_provider_item = formatText(ucfirst($lang['sdi']['p_title'])) ;

  $section->addTitle($out_provider_item, 2);
  
  $styleTable = array('cellMargin' => 50);
  $table = $section->addTable($styleTable);
  
  $fStyle = array();
  $clStyle = array('align' => 'right');
  $crStyle = array('align' => 'left');
  
  $row = $table->addRow();
  $row->addCell(PAGE_WIDTH * $lwidth)->addText('');
  $row->addCell(PAGE_WIDTH * (1 - $lwidth))->addText($out_name, $fStyle, $crStyle);
  
  $row = $table->addRow();
  $row->addCell(PAGE_WIDTH * $lwidth)->addText('');
  $row->addCell(PAGE_WIDTH * (1 - $lwidth))->addText($out_adress, $fStyle, $crStyle);
  
  $row = $table->addRow();
  $row->addCell(PAGE_WIDTH * $lwidth)->addText('');
  $row->addCell(PAGE_WIDTH * (1 - $lwidth))->addText($out_phone_fax, $fStyle, $crStyle);
  
  $row = $table->addRow();
  $row->addCell(PAGE_WIDTH * $lwidth)->addText('');
  $cell = $row->addCell(PAGE_WIDTH * (1 - $lwidth));
  $textrun = $cell->addTextRun();
  $textrun->addText($out_email_item, $fStyle);
	if (empty($out_email)) {
		$textrun->addText(empty_nc($out_email), $fStyle);
	} else {
		$textrun->addLink('mailto:' . $out_email, $out_email, $fStyle);
	}

	$section->addTextBreak(1);

  return true;
}

/**
 * DisplayReglementationInfo()
 * Affiche la réglementation d'un IDD
 *
 * @param  $current_record
 * @return bool true
 */
function DisplayReglementationInfo($current_record)
{
  global $section;
  global $lang;
  
  $lwidth = 0.29;

  $out_name = formatText(empty_nc($current_record['sdir_title'])) . PHP_EOL;
  $out_body = formatText(empty_nc(do_strip($current_record['sdir_body']))) . PHP_EOL;
  $out_uri = formatText($current_record['sdir_mask_uri']);
  $out_uri_href = $current_record['sdir_referer_uri'];

  $out_name_item = mb_ucfirst($lang['sdi']['r_title']) . ' : ';
  $out_body_item = mb_ucfirst($lang['sdi']['r_body']) . ' : ';
  $out_uri_item = mb_ucfirst($lang['sdi']['r_referer_uri']) . ' : ';

  $out_reglementation_item = formatText(ucfirst($lang['sdi']['add_step4'])) ;

  $section->addTitle($out_reglementation_item, 2);
  
  $styleTable = array('cellMargin' => 50);
  $table = $section->addTable($styleTable);

	$fStyle = array();
	$clStyle = array('align' => 'right');
	$crStyle = array('align' => 'left');

	// nom de l'organisme fournisseur
	$row = $table->addRow();
	$row->addCell(PAGE_WIDTH * $lwidth)->addText($out_name_item, 'softLabel', $clStyle);
	$row->addCell(PAGE_WIDTH  * (1 - $lwidth))->addText($out_name, $fStyle, $crStyle);

  // détail reglementation
	$row = $table->addRow();
	$row->addCell(PAGE_WIDTH * $lwidth)->addText($out_body_item, 'softLabel', $clStyle);
	$row->addCell(PAGE_WIDTH  * (1 - $lwidth))->addText($out_body, $fStyle, $crStyle);
	
  // lien internet reglementation
	$row = $table->addRow();
	$row->addCell(PAGE_WIDTH * $lwidth)->addText($out_uri_item, 'softLabel', $clStyle);
	if (!empty($out_uri_href)) {
		if (!empty($out_uri)) $row->addCell(PAGE_WIDTH  * (1 - $lwidth))->addLink($out_uri_href, $out_uri, $fStyle);
	} else {
		$row->addCell(PAGE_WIDTH  * (1 - $lwidth))->addText(empty_nc($out_uri), $fStyle, $crStyle);
	}
	
	$section->addTextBreak(1);
	
  return true;
}

/**
 * DisplayEvaluationInfo()
 * Affiche le tableau de critères d'évaluation
 *
 * @param int $current_record
 * @return bool true
 */
function DisplayEvaluationInfo($current_record)
{
  global $section;
  global $phpWord;
  global $lang;
  global $cellalign;
  global $styleTable;
  global $firstRowStyle;

  $out_eval_item = formatText(mb_ucfirst($lang['sdi']['e_title'])) ;


  $section->addTitle($out_eval_item, 2);
  
  $phpWord->addTableStyle('evaluation', $styleTable, $firstRowStyle);
  $table = $section->addTable('evaluation');
  
  $fStyle = array('size' => FONT_SIZE_MIN);
  $cStyle = array('align' => 'center');
  
  // header
  $row = $table->addRow();
  $row->addCell(PAGE_WIDTH / 6, $cellalign)->addText(mb_ucfirst($lang['sdi']['e_scale_compare']), 'arrayHeader', $cStyle);
  $row->addCell(PAGE_WIDTH / 6, $cellalign)->addText(mb_ucfirst($lang['sdi']['e_fiability']), 'arrayHeader', $cStyle);
  $row->addCell(PAGE_WIDTH / 6, $cellalign)->addText(mb_ucfirst($lang['sdi']['e_accessibility']), 'arrayHeader', $cStyle);
  $row->addCell(PAGE_WIDTH / 6, $cellalign)->addText(mb_ucfirst($lang['sdi']['e_lisibility']), 'arrayHeader', $cStyle);
  $row->addCell(PAGE_WIDTH / 6, $cellalign)->addText(mb_ucfirst($lang['sdi']['e_relevance']), 'arrayHeader', $cStyle);
  $row->addCell(PAGE_WIDTH / 6, $cellalign)->addText(mb_ucfirst($lang['sdi']['e_global_performance']), 'arrayHeader', $cStyle);
  
  // values
  $row = $table->addRow();
  $row->addCell(PAGE_WIDTH / 6, $cellalign)->addText($current_record['sdie_scale_compare'], $fStyle, $cStyle);
  $row->addCell(PAGE_WIDTH / 6, $cellalign)->addText($current_record['sdie_fiability'], $fStyle, $cStyle);
  $row->addCell(PAGE_WIDTH / 6, $cellalign)->addText($current_record['sdie_accessibility'], $fStyle, $cStyle);
  $row->addCell(PAGE_WIDTH / 6, $cellalign)->addText($current_record['sdie_lisibility'], $fStyle, $cStyle);
  $row->addCell(PAGE_WIDTH / 6, $cellalign)->addText($current_record['sdie_relevance'], $fStyle, $cStyle);
  $row->addCell(PAGE_WIDTH / 6, $cellalign)->addText($current_record['sdie_global_performance'], $fStyle, $cStyle);

  $section->addTextBreak(1);

  return true;
}


/**
 * DisplayValues()
 * Affiche les valeurs d'un IDD.
 *
 * @param int $sdi_id
 * @return bool true
 **/
function DisplayValues($sdi_id)
{
  global $section;
  global $lang;
  global $report_settings;
  global $scale_array;
  global $sql_object;
  global $scale_denomination_array;

  if(count($scale_array) == 0) return true;

  if ($report_settings['graphic_values'] === true && $report_settings['table_values'] === true) $mode = 'TWICE';
  elseif ($report_settings['graphic_values'] === true || $report_settings['table_values'] === true) $mode = 'ONE';
  else return false;

  $out_values_item = mb_ucfirst($lang['dashboard']['value']);
  
  $section->addTitle($out_values_item, 2);

  $scale_no_values = array(); // recipient no values
  $scale_insuffisant_values = array(); // recipient insuffisant values graph


  for($i = 0; $i < count($scale_array); $i++) {

  	// we print chart if asked
    if ($report_settings['graphic_values'] === true) {
      if (DisplayGraphicValues($sdi_id, $scale_array[$i], GetStatus($report_settings['status']), $mode) === false) array_push($scale_insuffisant_values, $scale_array[$i]);
    }

    // we print values if asked
    if ($report_settings['table_values'] === true) {
    	if (DisplayTableValues($sdi_id, $scale_array[$i], GetStatus($report_settings['status']), $mode) === false) array_push($scale_no_values, $scale_array[$i]);
    }

  }
  DisplayNoValuesFor($scale_no_values, $lang['dashboard']['rap_no_values_for']);
  DisplayNoValuesFor($scale_insuffisant_values, $lang['dashboard']['rap_insuffisant_values'], $mode, $scale_no_values);

  return true;
}


/**
 * GetStatus()
 * Formattage du statut pour requête SQL
 *
 * @param string $current_status
 * @return string $status
 **/
function GetStatus($current_status)
{
  switch ($current_status) {
    case 'ALL':
      $status = 'SCA';
      break;
    case 'PUBLIC':
      $status = 'SCP';
      break;
  }
  return $status;
}

/**
 * DisplayNoValuesFor()
 * Affiche les échelles ne contenant pas de valeurs ou insuffisamment
 *
 * @param array $array_scale
 * @param string $message
 * @param integer $mode
 * @param integer $array
 * @return bool true
 **/
function DisplayNoValuesFor($array_scale, $message, $mode = -1, $array = -1)
{
  global $section;
  global $lang;
  global $scale_denomination_array;
  $content = '';
  $sep = '';
  // si mode TWICE et deuxième tableau fourni. On le soustrait au premier.
  if ($array != -1 && $mode == 'TWICE') $array_scale = array_values(array_diff($array_scale, $array));

  if (count($array_scale) == 0) return true;

  for($i = 0; $i < count($array_scale); $i++) {
    $content .= $sep . $scale_denomination_array[$array_scale[$i]];
    $sep = ', ';
  }
  if (empty($content)) return true;
  
  $textrun = $section->addTextRun();
  $textrun->addText($message, 'noValueFor');
  $textrun->addText($content, array());
  $section->addTextBreak(1);

  return true;
}

/**
 * DisplayGraphicValues()
 * Gènère et affiche les valeurs sous forme graphique
 *
 * @param int $sdi_id
 * @param int $current_scale
 * @param string $status
 * @param string $mode
 * @return bool true
 **/
function DisplayGraphicValues($sdi_id, $current_scale, $status, $mode)
{
	global $section;
	global $sql_object;
	global $scale_denomination_array;
	

	include_once(override('../dashboard/graph.php'));
	
 	$values = $sql_object->DBSelect(SQL_getAllValue($status, $current_scale, $sdi_id));
	
// 	// renversement des valeurs pour affichage chronologique
// 	if(is_array($values)) $values = @array_reverse ($values, false);

// 	$result_sdii = $sql_object->DBSelect(SQL_getInfoSdi($sdi_id));

	if (count($values) < 3) 	return false;

	generateGraphic($sdi_id, $current_scale, $status, 'auto', false, '../tmp/report');
	
	// /////////////////////////////
	// Affichage
	
	// Affichage du titre
	$out_scale_item = $scale_denomination_array[$current_scale];
	
	$scaleStyle = array('bold' => true);
	$section->addText($out_scale_item, $scaleStyle);
	// $section->addTextBreak(1);
	
	// Affichage de l'image
	$section->addImage('../tmp/report/'.$sdi_id.'_'.$current_scale.'.png', array('align' => 'center', 'width' => 500));

	$section->addTextBreak(1);

	return true;
	
}

/**
 * DisplayTableValues()
 * Affiche un tableau de valeurs a une échelle donnée
 *
 * @param int $sdi_id
 * @param int $current_scale
 * @param string $status
 * @param string $mode
 * @return bool true
 **/
function DisplayTableValues($sdi_id, $current_scale, $status, $mode)
{
	
	global $section;
	global $phpWord;
	global $lang;
	global $sql_object;
	global $scale_denomination_array;
	global $cellalign;
	global $styleTable;
	global $firstRowStyle;

	$values = array();

	// setting $default_threshold value
	// By default initial value
	$result_sdii = $sql_object -> DBSelect(SQL_getInfoSdi($sdi_id));
	$default_threshold = $result_sdii[0]['sdii_threshold_value'];

	$sdi_values = $sql_object->DBSelect(SQL_getAllValue($status, $current_scale, $sdi_id));
	

	if (count($sdi_values) <= 1 && $sdi_values[0] == false) return false;

	// we prepare values for displaying
	for($i = 0; $i < count($sdi_values); $i++) {

		if(!empty($sdi_values[$i]['sdiv_multivalue'])) {

			$data = unserialize($sdi_values[$i]['sdiv_multivalue']);
			$current_value = $data;
			
		} else {
			
			$current_value = '';
			
			// If indicator is boolean
			if($result_sdii[0]['sdii_type'] == 'boolean') {
				$a = getBooleanValues($result_sdii[0]);
				if($a) {
					$current_value .= $a[$sdi_values[$i]['sdiv_value']];
				} else {
					$current_value .= fnumber_format($sdi_values[$i]['sdiv_value'], 'auto', false);
				}
				 
			} else {
				$current_value .= fnumber_format($sdi_values[$i]['sdiv_value'], 'auto', false);
			}
			
		}
		 
		// if set, we get it from the table
		if(!is_null($sdi_values[$i]['sdiv_threshold'])) {
			$threshold = fnumber_format($sdi_values[$i]['sdiv_threshold'], 'auto', false);
		} else {
			$threshold = fnumber_format($default_threshold, 'auto', false);
		}
		 
		if ($sdi_values[$i]['sdiv_statut'] == 'D') $current_value .= ' *';
		if (!empty($sdi_values[$i]['sdiv_comment'])) {
			if($sdi_values[$i]['sdiv_comment_display']=='Y') {
				$comment_status = _t('dashboard', 'public');
			} else {
				$comment_status = _t('dashboard', 'private');
			}
			
			$comments = formatText(do_strip($sdi_values[$i]['sdiv_comment'])) .' ('.$comment_status.')'.PHP_EOL;

		} else {
			$comments = '';
		}
		
		$current_date = formatText($sdi_values[$i]['date_p']);
		
		$values[$i]['current'] = $current_value;
		$values[$i]['threshold'] = $threshold;
		$values[$i]['date'] = $current_date;
		$values[$i]['comment'] = $comments;

	}

	// Affichage des données
	
	// Affichage du titre seulement si affichage des valeurs en standalone
	// ou si la génération des graphiques est infaisables car moins de 3 valeurs
	if ($mode != 'TWICE' || (count($sdi_values) >= 1 && count($sdi_values) < 3)) {
		$out_scale_item = $scale_denomination_array[$current_scale];
	
		$scaleStyle = array('bold' => true);
		$section->addText($out_scale_item, $scaleStyle);
	}
	
	$phpWord->addTableStyle('values', $styleTable, $firstRowStyle);
	$table = $section->addTable('values');
	
	$fStyle = array('size' => FONT_SIZE_MIN);
  $cStyle = array('align' => 'center');
  $clStyle = array('align' => 'left');

  
  // header
  $row = $table->addRow();
  $row->addCell(PAGE_WIDTH * 0.4, $cellalign)->addText(mb_ucfirst($lang['dashboard']['name_pluriel']), 'arrayHeader', $cStyle);
  $row->addCell(PAGE_WIDTH * 0.2, $cellalign)->addText(mb_ucfirst($lang['dashboard']['threshold_value']), 'arrayHeader', $cStyle);
  $row->addCell(PAGE_WIDTH * 0.2, $cellalign)->addText(mb_ucfirst($lang['statut']['published_on']), 'arrayHeader', $cStyle);
  $row->addCell(PAGE_WIDTH * 0.2, $cellalign)->addText(mb_ucfirst($lang['sdi']['comment']), 'arrayHeader', $cStyle);
	
  // values
	foreach ($values as &$val) {
		$row = $table->addRow();
		
		$cell = $row->addCell(PAGE_WIDTH * 0.4, $cellalign);
		
		if(is_array($val['current'])) {
			foreach($val['current'] as &$el) {
			
				$textrun = $cell->addTextRun();
				
				if(!empty($el['label'])) $disp = mb_ucfirst($el['label']). ' : ';
				$disp .= $el['value'];
				
				$textrun->addText($disp, 'arrayValue');
				$textrun->addText(' ('.fnumber_format($el['_percentage'], 2, false).' %)', 'percentageValue');

			}
			// we display the total
			$cell->addText(mb_ucfirst(_t('dashboard', 'multivalue_total')) . ' : ' . $val['current'][0]['_total'], array('size' => FONT_SIZE_MIN, 'bold' => true));
			

		} else {
			$cell->addText($val['current'], 'arrayValue', $cStyle);
		}
		
		$row->addCell(PAGE_WIDTH * 0.2, $cellalign)->addText($val['threshold'], $fStyle, $cStyle);
		$row->addCell(PAGE_WIDTH * 0.2, $cellalign)->addText($val['date'], $fStyle, $cStyle);
		$row->addCell(PAGE_WIDTH * 0.2, $cellalign)->addText($val['comment'], $fStyle, $clStyle);
	}
	
	
	// Affichage de l'astérisque - avertissement
	$section->addText('* ' . strtolower($lang['statut']['draftpdf']), array('size' => FONT_SIZE_MIN, 'italic' => true));

	$section->addTextBreak(1);
		
	return true;
}

/**
 * do_strip()
 * @param string $content
 * @return string
 */
function do_strip($content) {

	global $export_format;
	
// 	if($export_format == 'html') return $content;
// 	else { 
// 		$converted = new html2text($content);
// 		return $converted->get_text();
// 	}
	// 	return strip_tags($content);
	$converted = new html2text($content);
	return $converted->get_text();
}


?>