<?php
/**
 * Export module - pdf
 * 
 * Contains all functions to generate PDF file
 * 
 * @package linea21\utils\export
 * @author $Author$ - linea21 <info@linea21.com>
 * @version $Id$ 
 * @access public
 * @license http://opensource.org/licenses/gpl-3.0.html
 * Define, compose and generate a SDI (Sustainable Development Indicators) PDF report
 */

/********************************************************************************************
 *
* Following functions are used to generate projects reports
*
********************************************************************************************/





/**
 * project_display_info()
 * Affiche les infos générales d'un projet
 *
 * @param int $pointer
 * @return bool true
 */
function project_display_info($pointer)
{
	global $project;
	global $pdf;
	global $report_settings;
	global $current_root_level;
	global $bookmark_level;
	
	$current_record = $project[$pointer];
	
	$pdf->AddPage();
	// lien interne si actif
	if ($report_settings['summary'] === true) $pdf->SetLink($pointer+1);
	
	// we retrieve root level
	$levels = $GLOBALS['sql_object']->DBSelect(SQL_getLevelsList());
	$root_level = getLevelParent($levels, $current_record['level_id'], 0);
	
	// we display Finalities as chapters if asked
	if ($report_settings['finalities'] ===  true && $current_record['project_level_id'] !== 0) {
		
		if(is_array($root_level) && $root_level['id'] != $current_root_level) {
			project_display_root_level($root_level);
			$current_root_level = $root_level['id'];
		} elseif(!is_array($root_level)) {
			$pdf->Bookmark(_t('report', 'no-finality'), 0, -1);
		}
		
	}

	// titre du projet
	$out_title = formatText(mb_ucfirst($current_record['project_name']));
	$pdf->SetTextColorArray(getColor('SDI_TITLE_COLOR'));
	$pdf->SetFont(DOC_POLICE, SDI_TITLE_STYLE, SDI_TITLE_SIZE);
	$pdf->Bookmark(mb_ucfirst($out_title), $bookmark_level);
	$pdf->Write(DOC_HEIGHT, $out_title);
	
	// si non publié
	if ($current_record['project_statut'] == 'D') {
		$pdf->SetTextColorArray(getColor('DOC_ADVISE_COLOR'));
		$pdf->SetFont(DOC_POLICE, '', SDI_TITLE_SIZE);
		$out_status = ' / ' . strtolower(_t('statut', 'draftpdf'));
		$pdf->Write(DOC_HEIGHT, $out_status);
		$pdf->SetFont(DOC_POLICE, SDI_TITLE_STYLE, SDI_TITLE_SIZE);
		$pdf->SetTextColorArray(getColor('DOC_COLOR'));
	}

	$pdf->Ln(DOC_HEIGHT);


	$out_level = formatText('» ' . mb_ucfirst($current_record['level_label']) . ' : ' . $current_record['level_name'] );
	// root parent has been retrieved before
	if(is_array($root_level) && $root_level['id'] != $current_record['level_id']) $out_level .=  ' / '. formatText( mb_ucfirst($root_level['label']) . ' : ' . $root_level['name']). PHP_EOL;

	if(!empty($current_record['level_name'])) {
		$pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);
		$pdf->SetTextColorArray(getColor('SDI_TITLE_COLOR'));
		$pdf->Write(DOC_HEIGHT, $out_level);
		$pdf->SetY($pdf->GetY() + DOC_INTERVAL);
		$pdf->SetTextColorArray(getColor('DOC_COLOR'));
	}
	
	// parent project
	if($current_record['project_parent_id'] != 0) {
		project_display_parent($current_record['parent_project_name']);
	}
	
	// goal
	if(!empty($current_record['project_goal'])) {
		$pdf->SetY($pdf->GetY() + DOC_INTERVAL);
		$converted = new html2text($current_record['project_goal']);
		$out_goal = formatText($converted->get_text()) . PHP_EOL;
		$out_goal_item = formatText(mb_ucfirst(_t('project', 'goal')));
		$pdf->Bookmark($out_goal_item, ($bookmark_level+1), -1);
		project_format_item($out_goal_item);
		$pdf->MultiCell(0, DOC_HEIGHT, $out_goal, 0, 'L');
		$pdf->SetY($pdf->GetY() + DOC_INTERVAL);
	}
	
	// description
	//   $out_desc = formatText(empty_nc(strip_tags($current_record['project_description']))) . PHP_EOL;
	if(!empty($current_record['project_description'])) {
		$converted = new html2text($current_record['project_description']);
		$out_desc = formatText(empty_nc($converted->get_text())) . PHP_EOL;
		$out_desc_item = formatText(mb_ucfirst(_t('project', 'description')));
		$pdf->Bookmark($out_desc_item, ($bookmark_level+1), -1);
		project_format_item($out_desc_item);
		$pdf->MultiCell(0, DOC_HEIGHT, $out_desc, 0, 'L');
		$pdf->SetY($pdf->GetY() + DOC_INTERVAL);
	}
	
	// body
	//   $out_body = formatText(empty_nc(strip_tags($current_record['project_body']))) . PHP_EOL;
	$converted = new html2text($current_record['project_body']);
	$out_body = formatText(empty_nc($converted->get_text())) . PHP_EOL;
	$out_body_item = formatText(mb_ucfirst(_t('project', 'body')));
	$pdf->Bookmark($out_body_item, ($bookmark_level+1), -1);
	project_format_item($out_body_item);
	$pdf->MultiCell(0, DOC_HEIGHT, $out_body, 0, 'L');
	$pdf->SetY($pdf->GetY() + DOC_INTERVAL);
	
	project_complement($current_record);

	project_display_calendar($current_record);
	project_display_steering_budget($current_record);
	
	project_associated_indicators($current_record);
	if($report_settings['dashboard_values']) project_associated_dashboard($current_record);
	
	return true;
}


/**
 * project_display_steering_budget()
 * Affiche les tableaux contenant information 'animation / comité de pilotage'
 * et budget
 *
 * @param int $current_record
 * @return bool true
 */
function project_display_calendar($current_record)
{
	global $pdf;
	global $bookmark_level;

	ParagraphBreak();
	$out_cal_item = formatText(mb_ucfirst(_t('project','calendar_title')));
	$pdf->Bookmark($out_cal_item, ($bookmark_level+1), -1);
	project_format_item($out_cal_item);
	$pdf->SetY($pdf->GetY() + DOC_INTERVAL);

	$tbl = '<style>
			table {
  			border:1px solid #E2E2E2;
				color:#777777;
  		}
			.header {	
				color:#777777;
				background-color:#E2E2E2;
				font-size:7pt;
			}
		  td.br {
		    	border-right:1px solid #E2E2E2;
		  }
			td {
				text-align:center;
			}
			</style>
      <table cellspacing="0" cellpadding="5" nobr="true">
          <tr>
              <td class="header">'.mb_ucfirst(_t('project', 'begin_date')).'</td>
              <td class="header">'.mb_ucfirst(_t('project', 'estimated_date')).'</td>
              <td class="header">'.mb_ucfirst(_t('project', 'end_date')).'</td>
              <td class="header">'.mb_ucfirst(_t('project', 'progression')).'</td>
          </tr>
        <tr>
        	<td class="br">'.empty_nc($current_record['project_begin_date_display']).'</td>
        	<td class="br">'.empty_nc($current_record['project_estimated_date_display']).'</td>
        	<td class="br">'.empty_nc($current_record['project_end_date_display']).'</td>
        	<td>'.empty_nc($current_record['project_completed']).'% </td>
      	</tr>
      </table>';

	$pdf->SetFont(DOC_POLICE, DOC_STYLE, MIN_SIZE);
	$pdf->writeHTML($tbl, true, false, false, false, '');
	$pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);

	// project_completion($current_record);
}

/**
 * project_associated_indicators()
 * Affiche la liste des indicateurs associés
 *
 * @param int $current_record
 * @return bool true
 */
function project_associated_indicators($current_record)
{
	global $pdf;
	global $bookmark_level;

	ParagraphBreak();
	$out_indicators_item = formatText(mb_ucfirst(_t('project','sdi')));
	$pdf->Bookmark($out_indicators_item, ($bookmark_level+1), -1);
	project_format_item($out_indicators_item);
	$pdf->SetY($pdf->GetY() + DOC_INTERVAL);
	
	$query = SQL_getProjectSdi($current_record['project_id']);
	$data = $GLOBALS['sql_object']->DBSelect($query);
	
	$pdf->setListIndentWidth(20); //  margin-left not implemented into tcpdf yet
	
	if(isset($data[0]['sdii_id'])) {
	  $sdi_list='<ul style="margin-left:10%;">'; //  margin-left not implemented into tcpdf yet
	  foreach ($data as &$value) {
	    $sdi_list.= '<li>'.formatText($value['sdii_name'], '2HTML').'</li>'.PHP_EOL;
	  }
	  $sdi_list.='</ul>';
	} else $sdi_list = mb_ucfirst(_t('divers','none'));

	$pdf->SetFont(DOC_POLICE, DOC_STYLE, MIN_SIZE);
	$pdf->writeHTML(empty_nc($sdi_list), true, false, false, false, '');
	$pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);

	ParagraphBreak();

	return true;
}

/**
 * project_associated_dashboard()
 * Affiche l'image des dernières valeurs des indicateurs
 *
 * @param int $current_record
 * @return bool true
 */
function project_associated_dashboard($current_record)
{
	global $pdf;
	
	if(file_exists(SITE_PATH. 'tmp/report/project-'.$current_record['project_id'].'-dashboard.png')) {
		$pdf->Image(SITE_PATH. 'tmp/report/project-'.$current_record['project_id'].'-dashboard.png',$pdf->GetX(), $pdf->GetY(), 135, 0, '', '', 'M', false, 0, 'C');
		$pdf->SetY($pdf->GetY() + DOC_INTERVAL);
	}

	return true;
}

/**
 * project_complement()
 * Affiche la priorité et l'échelle de rattachement
 *
 * @param int $current_record
 * @return bool true
 */
function project_complement($current_record)
{
	global $pdf;

	// ParagraphBreak();
	
	$complement = '<span class="label">' . mb_ucfirst(_t('project', 'statut')). ' : </span>' . display_statut($current_record['project_statut']);
	
	
	$complement .= '<span class="separator"> / </span><span class="label">' . mb_ucfirst(_t('project', 'priority')). ' : </span>' .$current_record['priority_name'] ;
	
	($current_record['scale_id'] != 0 ) ? $complement .=  '<span class="separator"> / </span><span class="label">' .mb_ucfirst(_t('project', 'scale')). ' : </span>' .$current_record['scale_denomination'] : '';

	$tbl = '<style>
			.complement {
				color:#444444;
				font-size:8pt;
				text-align:right;
			}
			span.label, span.separator {
				color:#777777;
				font-weight:bold,
				}	
			</style>
			<div class="complement">'.$complement.'</div>
      ';

	$pdf->SetFont(DOC_POLICE, DOC_STYLE, MIN_SIZE);
	$pdf->writeHTML($tbl, true, false, false, false, '');
	$pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);

	return true;
}

/**
 * project_completion()
 * Affiche un indcateur de progression de l'action
 *
 * @param int $current_record
 * @return bool true
 * @todo complete - it dose not work yet ... not used
 */
function project_completion($current_record)
{
	global $pdf;

	ParagraphBreak();

	$totalwidth = 700;
	$currentwidth = $current_record['project_completed'] * $totalwidth / 100 ;


	$tbl = '<style>
			td.value {
		    	width:70%;
		  }
		  td.label {
		    	width:30%;
		  }
			td {
		    	text-align:left;
		  }
			</style>
      <table cellspacing="0" cellpadding="0" nobr="true">
          <tr>
              <td class="label">'.mb_ucfirst(_t('project','progression')) .' : </td>
              <td class="value" style="background-color:#E5E5E5"><div style="width:'.$currentwidth.'px;background-color:#FCA874;display:block;color:white;text-align:right;">' . $current_record['project_completed'] . ' % </div></td>
          </tr>
      </table>';

	$pdf->SetFont(DOC_POLICE, DOC_STYLE, MIN_SIZE);
	$pdf->writeHTML($tbl, true, false, false, false, '');
	$pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);

	return true;
}

/**
 * project_display_steering_budget()
 * Affiche les tableaux contenant information 'animation / comité de pilotage' 
 * et budget
 *
 * @param int $current_record
 * @return bool true
 */
function project_display_steering_budget($current_record)
{
	global $pdf;
	global $bookmark_level;

	ParagraphBreak();
	$pdf->Bookmark(_t('project','steering_title'), ($bookmark_level+1), -1);
	$pdf->Bookmark(_t('project','budget_title'), ($bookmark_level+1), -1);
	$pdf->SetX($pdf->GetX());
	
	$coordinator = getAssociatedActors($current_record['project_id'], 'coordinator');
	$partners = getAssociatedActors($current_record['project_id'], 'partner');
	$manager = getAssociatedManagers($current_record['project_id']);


	$tbl = '<style>
			.light {
				border-top:1px solid #444444;
			}
		    td.center {
		    	text-align:center;
		    }
			 td {
		    	text-align:left;
		    }
			td.title {
				font-weight:bold;
				color:#444444;
			}
			span.label {
				color:#777777;
				font-weight:bold,
				}	
			</style>
      <table cellspacing="5" cellpadding="5" nobr="true">
          <tr>
              <td class="center title">'.strtoupper(mb_ucfirst(_t('project', 'steering_title'))).'</td>
              <td class="center title">'.strtoupper(mb_ucfirst(_t('project', 'budget_title'))).'</td>
          </tr>
        <tr>
        	<td  class="light">		
              		<!-- Steering -->
              		<table cellspacing="0" cellpadding="5" border="0" nobr="true">
              			<tr>
              				<td><span class="label">'. mb_ucfirst(_t('project','elected')) . ' : </span>' . empty_nc($current_record['project_elected']) . '</td>			
              			</tr>
              			<tr>
              				<td><span class="label">'. mb_ucfirst(_t('project','coordinator')) . ' : </span>' . empty_nc($coordinator) . '</td>
              			</tr>
              			<tr>	              				
              				<td><span class="label">'. mb_ucfirst(_t('project','manager')) . ' : </span>' . empty_nc($manager) . '</td>
              			</tr>
              			<tr>
              				<td><span class="label">'. mb_ucfirst(_t('project','partners')) . ' : </span>' . empty_nc($partners) . '</td>
              			</tr>
              			<tr>
              				<td><span class="label">'. mb_ucfirst(_t('project','team')) . ' : </span>' . empty_nc($current_record['project_team']) . '</td>
              			</tr>		
              		</table>
           </td>
           <td class="light">
              			<!-- Budget -->
              		<table cellspacing="0" cellpadding="5" border="0" nobr="true">
              			<tr>
              				<td><span class="label">'.  mb_ucfirst(_t('project','budget')) . ' : </span>'.  empty_nc(fnumber_format($current_record['project_budget'],0, false)) . '</td>			
              			</tr>
              			<tr>
              				<td><p><span class="label">'.  mb_ucfirst(_t('project','budget_comment')) . ' : </span></p><div>' . empty_nc($current_record['project_budget_comment']) . '</div></td>
              			</tr>	
              		</table>			
           </td>

      	</tr>
      </table>';

	$pdf->SetFont(DOC_POLICE, DOC_STYLE, MIN_SIZE);
	$pdf->writeHTML($tbl, true, false, false, false, '');
	$pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);

	return true;
}

function project_display_parent($name)
{
	global $pdf;
	
	$converted = new html2text($name);
	$out_parent = formatText($converted->get_text()) . PHP_EOL;
	$out_parent_item = formatText(mb_ucfirst(_t('project', 'parent')));
	$pdf->Ln();
	$pdf->writeHTML('<div style="font-size: 8pt;color:#888888"><strong>'. $out_parent_item . '</strong> : '  . $out_parent . '</div>', true, 0, true, true);
	$pdf->Ln();

}

function project_display_root_level($level)
{
	global $pdf;
	global $report_settings;
	
	ParagraphBreak();
	ParagraphBreak();
	
	$pdf->SetFont(DOC_POLICE, SDI_TITLE_STYLE, SDI_TITLE_SIZE);
	$pdf->SetTextColorArray(getColor('SDI_TITLE_COLOR'));
	$pdf->MultiCell(0,DOC_HEIGHT,mb_ucfirst($level['name']),0,'L',0);
	$pdf->SetTextColorArray(getColor('DOC_COLOR'));
	$pdf->SetFont(DOC_POLICE,DOC_STYLE,DOC_SIZE);
	$pdf->Bookmark(mb_ucfirst($level['name']), 0, -1);
	
	ParagraphBreak();
	
	if ($report_settings['finalities_description'] === true) {
		
		$converted = new html2text($level['main_desc']);
		$out = formatText($converted->get_text()) . PHP_EOL;
		$pdf->Ln();
		$pdf->writeHTML('<div style="font-size: 10pt;color:#777777;text-align:justify">'. $out . '</div>', true, 0, true, true);
		$pdf->writeHTML('<br pagebreak="true" />');

	}

}

function project_format_item($libelle)
{
	global $pdf;

	$libelle="        ".$libelle;
	$pdf->SetFont(DOC_POLICE, SDI_ITEM_STYLE, SDI_ITEM_SIZE);
	$pdf->SetTextColorArray(getColor('SDI_ITEM_COLOR'));
	$pdf->MultiCell(0,DOC_HEIGHT,$libelle,0,'L',0);
	$pdf->SetTextColorArray(getColor('DOC_COLOR'));
	$pdf->SetFont(DOC_POLICE,DOC_STYLE,DOC_SIZE);
}




/********************************************************************************************
 * 
 * Following functions are used to generate dashboard reports
 * 
 ********************************************************************************************/

/**
 * indicator_display_info()
 * Affiche un IDD + infos satellites
 *
 * @param int $pointer
 * @return bool true
 */
function indicator_display_info($pointer)
{
  global $sdi;
  global $pdf;
  global $report_settings;

  $pdf->AddPage();
  // lien interne si actif
  if ($report_settings['summary'] === true) $pdf->SetLink($pointer+1);
  indicator_display_main($sdi[$pointer]);
  indicator_display_values($sdi[$pointer]['sdii_id']);
  if ($report_settings['display_provider'] === true) indicator_display_provider($sdi[$pointer]);
  if ($report_settings['display_reglementation'] === true) indicator_display_rules($sdi[$pointer]);
  if ($report_settings['display_evaluation'] === true) indicator_display_assessment($sdi[$pointer]);

  return true;
}


function indicator_format_item($libelle)
{
  global $pdf;

  $libelle="        ".$libelle;
  $pdf->SetFont(DOC_POLICE, SDI_ITEM_STYLE, SDI_ITEM_SIZE);
  $pdf->SetTextColorArray(getColor('SDI_ITEM_COLOR'));
  $pdf->MultiCell(0,DOC_HEIGHT,$libelle,0,'L',0);
  $pdf->SetTextColorArray(getColor('DOC_COLOR'));
  $pdf->SetFont(DOC_POLICE,DOC_STYLE,DOC_SIZE);
}

function indicator_format_indic($libelle)
{
  global $pdf;

  $pdf->SetFillColorArray(getColor('BCKG_INFO'));
  $pdf->SetFont(DOC_POLICE, SDI_INFO_STYLE, SDI_INFO_SIZE);
  $pdf->SetTextColorArray(getColor('SDI_INFO_COLOR'));
  $pdf->MultiCell(0,DOC_HEIGHT+2,'   '.$libelle,0,'L',1, 1, '', '', true, 0, false, true, DOC_HEIGHT+2, 'M');
//   $pdf->SetLineStyle(array('color'=> array(100,100,100)));
//   $pdf->Line($pdf->GetX(), $pdf->GetY(), $pdf->GetX()+180, $pdf->GetY());
  $pdf->SetTextColorArray(getColor('DOC_COLOR'));
  $pdf->SetFont(DOC_POLICE,DOC_STYLE,DOC_SIZE);
  $pdf->Ln(DOC_INTERVAL);
}

// Saut de paragraphe
function ParagraphBreak()
{
  global $pdf;

  $pdf->SetTextColorArray(getColor('DOC_COLOR'));
  $pdf->SetFont(DOC_POLICE,DOC_STYLE,DOC_SIZE);
  $pdf->Ln(PARAGRAPH_BREAK);
}

/**
 * indicator_display_main()
 * Affiche les infos générales d'un IDD
 *
 * @param int $current_record
 * @return bool true
 */
function indicator_display_main($current_record)
{
  global $pdf;
  global $report_settings;

  // titre d'un indicateur
  $out_title = formatText($current_record['sdii_name']);
  $pdf->SetTextColorArray(getColor('SDI_TITLE_COLOR'));
  $pdf->SetFont(DOC_POLICE, SDI_TITLE_STYLE, SDI_TITLE_SIZE);
  $pdf->Bookmark(mb_ucfirst(strtolower($out_title)));
  $pdf->Write(DOC_HEIGHT, $out_title);
  // si non publié
  if ($current_record['sdii_statut'] == 'D') {
    $pdf->SetTextColorArray(getColor('DOC_ADVISE_COLOR'));
    $pdf->SetFont(DOC_POLICE, '', SDI_TITLE_SIZE);
    $out_status = ' / ' . strtolower(_t('statut', 'draftpdf'));
    $pdf->Write(DOC_HEIGHT, $out_status);
    $pdf->SetFont(DOC_POLICE, SDI_TITLE_STYLE, SDI_TITLE_SIZE);
    $pdf->SetTextColorArray(getColor('DOC_COLOR'));
  }
  // si non attaché au tableau de bord
  if ($current_record['sdii_to_dashboard'] == 'N') {
  	$pdf->SetTextColorArray(getColor('DOC_ADVISE_COLOR'));
  	$pdf->SetFont(DOC_POLICE, '', SDI_TITLE_SIZE);
  	$out_attached_to_dashboard = ' (' . strtolower(_t('sdi','not_attached_to_dashboard').')');
  	$pdf->Write(DOC_HEIGHT, $out_attached_to_dashboard);
  	$pdf->SetFont(DOC_POLICE, SDI_TITLE_STYLE, SDI_TITLE_SIZE);
  	$pdf->SetTextColorArray(getColor('DOC_COLOR'));
  }

  $pdf->Ln(DOC_HEIGHT);
  // thème d'appartenance

  $out_level = formatText('» ' . mb_ucfirst(_t('sdi', 'level')) . ' : ' . $current_record['level_name'] );
  // we retrieve root level
  $levels = $GLOBALS['sql_object']->DBSelect(SQL_getLevelsList());
  $a = getLevelParent($levels, $current_record['level_id'], 0);
  if(is_array($a) && $a['id'] != $current_record['level_id']) $out_level .=  ' / '. formatText( mb_ucfirst($a['label']) . ' : ' . $a['name']). PHP_EOL;
  
  $pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);
  $pdf->SetTextColorArray(getColor('SDI_TITLE_COLOR'));
  $pdf->Write(DOC_HEIGHT, $out_level);
  $pdf->SetY($pdf->GetY() + DOC_INTERVAL);
  $pdf->SetTextColorArray(getColor('DOC_COLOR'));
  // description
  $out_desc = formatText(strip_tags($current_record['sdii_description'])) . PHP_EOL;
	$converted = new html2text($current_record['sdii_description']);
  $out_desc = formatText($converted->get_text()) . PHP_EOL;
  $pdf->Bookmark(mb_ucfirst(_t('sdi','description')), 1, -1);
  $pdf->MultiCell(0, DOC_HEIGHT, $out_desc, 0, 'L');
  $pdf->SetY($pdf->GetY() + DOC_INTERVAL);
  // objectif
//   $out_goal = formatText(empty_nc(strip_tags($current_record['sdii_goal']))) . PHP_EOL;
  $converted = new html2text($current_record['sdii_goal']);
  $out_goal = formatText(empty_nc($converted->get_text())) . PHP_EOL;
  $out_goal_item = formatText(mb_ucfirst(_t('sdi', 'goal')));
  $pdf->Bookmark($out_goal_item, 1, -1);
  indicator_format_item($out_goal_item);
  $pdf->MultiCell(0, DOC_HEIGHT, $out_goal, 0, 'L');
  $pdf->SetY($pdf->GetY() + DOC_INTERVAL);
  // conseil d'amélioration
//   $out_consulting = formatText(empty_nc(strip_tags($current_record['sdii_consulting']))) . PHP_EOL;
  $converted = new html2text($current_record['sdii_consulting']);
  $out_consulting = formatText(empty_nc($converted->get_text())) . PHP_EOL;
  $out_consulting_item = formatText(mb_ucfirst(_t('sdi', 'consulting')));
  $pdf->Bookmark($out_consulting_item, 1, -1);
  indicator_format_item($out_consulting_item);
  $pdf->MultiCell(0, DOC_HEIGHT, $out_consulting, 0, 'L');
  $pdf->SetY($pdf->GetY() + DOC_INTERVAL);


  // informations de mesure
  $out_mesures_item = formatText(ucfirst(_t('sdi', 'info_mesure')));
  $out_mesures_behavior = ($current_record['sdii_threshold_relative'] == 'Y' ? _t('sdi', 'threshold_relative_Y') : _t('sdi', 'threshold_relative_N'));
  $out_mesures_unit = formatText(mb_ucfirst(_t('sdi', 'unit'))) . ' : ' . $current_record['sdii_unit'] . PHP_EOL;
  $pdf->Bookmark($out_mesures_item, 1, -1);
  indicator_format_item($out_mesures_item);
  // unitée de mesure
  $pdf->SetFont(DOC_POLICE, EM_STYLE, MIN_SIZE);
  $pdf->Write(DOC_HEIGHT, $out_mesures_unit . PHP_EOL);
  // comportement
  $pdf->Write(DOC_HEIGHT, mb_ucfirst($out_mesures_behavior) . PHP_EOL);
  $pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);
  // tableau de valeurs
  indicator_display_mesures($current_record);
  $pdf->SetFont(DOC_POLICE, EM_STYLE, MIN_SIZE);
  $pdf->SetY($pdf->GetY() + DOC_INTERVAL);
}

/**
 * indicator_display_mesures()
 * Dessine le tableau des valeurs IDD
 *
 * @param  $current_record
 * @return bool true
 */
function indicator_display_mesures($current_record)
{

  global $pdf;
  
  $a = getBooleanValues($current_record);
  
  // if indicator is boolean or qualitative and mask is defined
  if(($current_record['sdii_type'] == 'boolean' || $current_record['sdii_type'] == 'qualitative') && $a) {
  	
  	if($a) {
  		$index = array();
  		foreach ($a as $key => $value) {
  			array_push($index, $key);
  		}
  	
  		$display_minvalue = $a[min($index)];
  		$display_maxvalue = $a[max($index)];
  		($current_record['sdii_threshold_value'] != '') ? $display_threshold = $a[$current_record['sdii_threshold_value']] : $display_threshold = empty_none('');
  	}

  } else {
  	$display_minvalue = empty_nc(fnumber_format($current_record['sdii_min_value'], 'auto', false));
  	$display_maxvalue = empty_nc(fnumber_format($current_record['sdii_max_value'], 'auto', false));
  	$display_threshold = empty_none(fnumber_format($current_record['sdii_threshold_value'], 'auto', false));
  }

  $tbl = '<style>
  	table {
  		border:1px solid #E2E2E2;
  		}
  	.header {
  		color:#777777;
			background-color:#E2E2E2;
  		font-weight:bold;
  		font-size:7pt;
  	}
    td {
    	text-align:center;
  		color:#777777;
    }
  	td.br {
  		border-right:1px solid #E2E2E2;
  		}
	</style>
  <table cellspacing="0" cellpadding="5" border="0" nobr="true">
  <thead>
      <tr>
          <td class="header">'.mb_ucfirst(_t('sdi', 'min_value')).'</td>
          <td class="header">'.mb_ucfirst(_t('sdi', 'max_value')).'</td>
          <td class="header">'.mb_ucfirst(_t('sdi', 'initial_threshold_value')).'</td>
          <td class="header">'.mb_ucfirst(_t('sdi', 'frequency')).'</td>
      </tr>
  </thead>
      <tr>
          <td class="br">'.$display_minvalue.'</td>
          <td class="br">'.$display_maxvalue.'</td>
          <td class="br">'.$display_threshold.'</td>
          <td>'.$current_record['sdii_frequency'].'</td>
      </tr>
  </table>';

  $pdf->SetFont(DOC_POLICE, DOC_STYLE, MIN_SIZE);
  $pdf->writeHTML($tbl, true, false, false, false, '');
  $pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);

  return true;
}

/**
 * indicator_display_provider()
 * Affiche les informations Fournisseurs d'un IDD
 *
 * @param  $current_record
 * @return bool true
 */
function indicator_display_provider($current_record)
{
  global $pdf;

  $out_name = formatText($current_record['sdip_name']);
  if (!empty($current_record['sdip_service'])) $out_name .= ' - ' . formatText($current_record['sdip_service']);
  if (!empty($current_record['sdip_incharge'])) $out_name .= ' - ' . formatText($current_record['sdip_incharge']) . ' (' . _t('sdi', 'p_incharge') . ')';
  $out_name .= PHP_EOL;
//   $out_adress = formatText(empty_nc(strip_tags($current_record['sdip_address'])));
  $converted = new html2text($current_record['sdip_address']);
  $out_adress = formatText(empty_nc($converted->get_text()));
  $out_phone_fax = mb_ucfirst(_t('sdi', 'p_phone')) . ' : ' . formatText(empty_nc($current_record['sdip_phone']));
  $out_phone_fax .= ' - ' . mb_ucfirst(_t('sdi', 'p_fax')) . ' : ' . formatText(empty_nc($current_record['sdip_fax'])) . PHP_EOL;
  $out_email_item = mb_ucfirst(_t('sdi', 'p_email')) . ' : ';
  $out_email = formatText($current_record['sdip_email']);
//   $out_desc = formatText(empty_nc(strip_tags($current_record['sdip_description']))) . PHP_EOL;
  $converted = new html2text($current_record['sdip_description']);
  $out_desc = formatText(empty_nc($converted->get_text())) . PHP_EOL;
  $out_provider_item = formatText(ucfirst(_t('sdi', 'p_title'))) ;

  ParagraphBreak();
  $pdf->Bookmark($out_provider_item, 1, -1);
  indicator_format_indic($out_provider_item);

  $firstcell_width = 60;

  $pdf->SetX($pdf->GetX() + $firstcell_width);
  $pdf->MultiCell(0, DOC_HEIGHT, $out_name, 0, 'L');

  $pdf->SetX($pdf->GetX() + $firstcell_width);
  $pdf->MultiCell(0, DOC_HEIGHT, $out_adress, 0, 'L');

  $pdf->SetX($pdf->GetX() + $firstcell_width);
  $pdf->MultiCell(0, DOC_HEIGHT, $out_phone_fax, 0, 'L');

  $pdf->SetX($pdf->GetX() + $firstcell_width);
  $pdf->Cell($pdf->GetStringWidth($out_email_item), DOC_HEIGHT, $out_email_item);
  if (empty($out_email)) $pdf->Cell(0, DOC_HEIGHT, empty_nc($out_email));
  else $pdf->Cell(0, DOC_HEIGHT, $out_email, 0, 0, 'L', 0, 'mailto:' . $out_email);
  $pdf->SetY($pdf->GetY() + DOC_INTERVAL);

  return true;
}

/**
 * indicator_display_rules()
 * Affiche la réglementation d'un IDD
 *
 * @param  $current_record
 * @return bool true
 */
function indicator_display_rules($current_record)
{
  global $pdf;

  $out_name = formatText(empty_nc($current_record['sdir_title'])) . PHP_EOL;
//    $out_body = formatText(empty_nc(strip_tags($current_record['sdir_body']))) . PHP_EOL;
  $converted = new html2text($current_record['sdir_body']);
  $out_body = formatText(empty_nc($converted->get_text())) . PHP_EOL;
  $out_uri = formatText($current_record['sdir_mask_uri']);
  $out_uri_href = $current_record['sdir_referer_uri'];

  $out_name_item = mb_ucfirst(_t('sdi', 'r_title')) . ' : ';
  $out_body_item = mb_ucfirst(_t('sdi', 'r_body')) . ' : ';
  $out_uri_item = mb_ucfirst(_t('sdi', 'r_referer_uri')) . ' : ';

  $out_reglementation_item = formatText(ucfirst(_t('sdi', 'add_step4'))) ;

  ParagraphBreak();
  $pdf->Bookmark($out_reglementation_item, 1, -1);
  indicator_format_indic($out_reglementation_item);

  $firstcell_width = 60;
  // nom de l'organisme fournisseur
  $pdf->SetFont(DOC_POLICE, EM_STYLE, DOC_SIZE);
  $pdf->Cell($firstcell_width, DOC_HEIGHT, $out_name_item, 0, 0, 'R');
  $pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);
  $pdf->MultiCell(0, DOC_HEIGHT, $out_name, 0, 'L');
  $pdf->SetY($pdf->GetY() + DOC_INTERVAL);
  // détail reglementation
  $pdf->SetFont(DOC_POLICE, EM_STYLE, DOC_SIZE);
  $pdf->Cell($firstcell_width, DOC_HEIGHT, $out_body_item, 0, 0, 'R');
  $pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);
  $pdf->MultiCell(0, DOC_HEIGHT, $out_body, 0, 'L');
  $pdf->SetY($pdf->GetY() + DOC_INTERVAL);
  // lien internet reglementation
  $pdf->SetFont(DOC_POLICE, EM_STYLE, DOC_SIZE);
  $pdf->Cell($firstcell_width, DOC_HEIGHT, $out_uri_item, 0, 0, 'R');
  $pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);
  if (!empty($out_uri_href)) {
    if (!empty($out_uri)) $pdf->Cell(0, DOC_HEIGHT, $out_uri, 0, 0, '', 0, $out_uri_href);
    //if (!empty($out_uri)) $pdf->Cell(0, DOC_HEIGHT, ' (', 0, 0);
    //$pdf->Cell(0, DOC_HEIGHT, $out_uri, 0, 0, '', 0, $out_uri_href);
    //if (!empty($out_uri)) $pdf->Cell(0, DOC_HEIGHT, ')', 0, 0);
  } else $pdf->Write(DOC_HEIGHT, empty_nc($out_uri));
  $pdf->SetY($pdf->GetY() + DOC_INTERVAL);

  return true;
}

/**
 * indicator_display_assessment()
 * Affiche le tableau de critères d'évaluation
 *
 * @param int $current_record
 * @return bool true
 */
function indicator_display_assessment($current_record)
{
  global $pdf;

  $out_eval_item = formatText(mb_ucfirst(_t('sdi', 'e_title'))) ;

  ParagraphBreak();
  $pdf->Bookmark($out_eval_item, 1, -1);
  indicator_format_indic($out_eval_item);
  $pdf->SetX($pdf->GetX());


  $tbl = '<style>
  		table {
  			border:1px solid #E2E2E2;
  		}
	  	.header {
	  		color:#777777;
				background-color:#E2E2E2;
	  		font-weight:bold;
	  		font-size:7pt;
	  	}
	    td {
	    	text-align:center;
	  		color:#777777;
	    }
	  	td.br {
	  		border-right:1px solid #E2E2E2;
	  		}
			</style>
      <table cellspacing="0" cellpadding="5" border="0" nobr="true">
          <tr>
              <td class="header">'.mb_ucfirst(_t('sdi', 'e_scale_compare')).'</td>
              <td class="header">'.mb_ucfirst(_t('sdi', 'e_fiability')).'</td>
              <td class="header">'.mb_ucfirst(_t('sdi', 'e_accessibility')).'</td>
              <td class="header">'.mb_ucfirst(_t('sdi', 'e_lisibility')).'</td>
              <td class="header">'.mb_ucfirst(_t('sdi', 'e_relevance')).'</td>
              <td class="header">'.mb_ucfirst(_t('sdi', 'e_global_performance')).'</td>
          </tr>
        <tr>
        		<td class="br">'.$current_record['sdie_scale_compare'].'</td>
            <td class="br">'.$current_record['sdie_fiability'].'</td>
            <td class="br">'.$current_record['sdie_accessibility'].'</td>
            <td class="br">'.$current_record['sdie_lisibility'].'</td>
            <td class="br">'.$current_record['sdie_relevance'].'</td>
            <td class="br">'.$current_record['sdie_global_performance'].'</td>
      	</tr>
        </tbody>
      </table>';

  $pdf->SetFont(DOC_POLICE, DOC_STYLE, MIN_SIZE);
  $pdf->writeHTML($tbl, true, false, false, false, '');
  $pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);

  return true;
}


/**
 * indicator_display_values()
 * Affiche les valeurs d'un IDD.
 *
 * @param int $sdi_id
 * @return bool true
 **/
function indicator_display_values($sdi_id)
{
  global $pdf;
  global $report_settings;
  global $scale_array;
  global $sql_object;
  global $scale_denomination_array;

  if(count($scale_array) == 0) return true;

  if ($report_settings['graphic_values'] === true && $report_settings['table_values'] === true) $mode = 'TWICE';
  elseif ($report_settings['graphic_values'] === true || $report_settings['table_values'] === true) $mode = 'ONE';
  else return false;

  $out_values_item = mb_ucfirst(_t('dashboard', 'all_value'));
  ParagraphBreak();
  $pdf->Bookmark($out_values_item, 1, -1);
  indicator_format_indic($out_values_item);

  $scale_no_values = array(); // recipient no values
  $scale_insuffisant_values = array(); // recipient insuffisant values graph


  for($i = 0; $i < count($scale_array); $i++) {

  	// we print chart if asked
    if ($report_settings['graphic_values'] === true) {
      if (indicator_display_graphic_values($sdi_id, $scale_array[$i], indicator_get_status($report_settings['status']), $mode) === false) array_push($scale_insuffisant_values, $scale_array[$i]);
    }

    // we print values if asked
    if ($report_settings['table_values'] === true) {
    	if (indicator_table_values($sdi_id, $scale_array[$i], indicator_get_status($report_settings['status']), $mode) === false) array_push($scale_no_values, $scale_array[$i]);
    }

  }
  indicator_no_values_for($scale_no_values, _t('report', 'no_values_for'));
  indicator_no_values_for($scale_insuffisant_values, _t('report', 'insuffisant_values'), $mode, $scale_no_values);

  return true;
}


/**
 * indicator_get_status()
 * Formattage du statut pour requête SQL
 *
 * @param string $current_status
 * @return string $status
 **/
function indicator_get_status($current_status)
{
  switch ($current_status) {
    case 'ALL':
      $status = 'SCA';
      break;
    case 'PUBLIC':
      $status = 'SCP';
      break;
  }
  return $status;
}

/**
 * indicator_no_values_for()
 * Affiche les échelles ne contenant pas de valeurs ou insuffisamment
 *
 * @param array $array_scale
 * @param string $message
 * @param integer $mode
 * @param integer $array
 * @return bool true
 **/
function indicator_no_values_for($array_scale, $message, $mode = -1, $array = -1)
{
  global $pdf;
  global $scale_denomination_array;
  $content = '';
  $sep = '';
  // si mode TWICE et deuxième tableau fourni. On le soustrait au premier.
  if ($array != -1 && $mode == 'TWICE') $array_scale = array_values(array_diff($array_scale, $array));

  if (count($array_scale) == 0) return true;

  for($i = 0; $i < count($array_scale); $i++) {
    $content .= $sep . $scale_denomination_array[$array_scale[$i]];
    $sep = ', ';
  }
  if (empty($content)) return true;
  $margin = 0;
  $pdf->SetX($pdf->GetX() + $margin);
  $pdf->SetFont(DOC_POLICE, EM_STYLE, DOC_SIZE);
  $pdf->Cell(0, DOC_HEIGHT, $message, 0, 1, 'L');
  $pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);
  $margin = 10;
  $pdf->SetTextColorArray(getColor('DOC_ADVISE_COLOR'));
  $pdf->SetX($pdf->GetX() + $margin);
  $pdf->MultiCell(0, DOC_HEIGHT, $content, 0, 'L');
  $pdf->SetY($pdf->GetY() + DOC_INTERVAL);
  $pdf->SetTextColorArray(getColor('DOC_COLOR'));

  return true;
}

/**
 * indicator_display_graphic_values()
 * Gènère et affiche les valeurs sous forme graphique
 *
 * @param int $sdi_id
 * @param int $current_scale
 * @param string $status
 * @param string $mode
 * @return bool true
 **/
function indicator_display_graphic_values($sdi_id, $current_scale, $status, $mode)
{
	global $pdf;
	global $sql_object;
	global $scale_denomination_array;
	

	include_once(override('../dashboard/graph.php'));
	

	// We retrieve indicator type to see if we generate graph or not
	$result_sdii = $sql_object -> DBSelect(SQL_getInfoSdi($sdi_id));
	$indicator_type = $result_sdii[0]['sdii_type'];
	
	
 	$values = $sql_object->DBSelect(SQL_getAllValue($status, $current_scale, $sdi_id));
	
// 	// renversement des valeurs pour affichage chronologique
// 	if(is_array($values)) $values = @array_reverse ($values, false);

// 	$result_sdii = $sql_object->DBSelect(SQL_getInfoSdi($sdi_id));

	if (count($values) < 2) 	return false;

	if($indicator_type == 'quantitative') {
		
		generateGraphic($sdi_id, $current_scale, $status, 'auto', false, '../tmp/report');
		
		// /////////////////////////////
		// Affichage
	
		// définition du positionnement
		$_y = $pdf->GetY() + 5;
		$_x = 60;
	
		// Affichage du titre
		$out_scale_item = $scale_denomination_array[$current_scale];
		//$pdf->Bookmark($out_scale_item, 2, -1);
		indicator_format_item($out_scale_item);
		$pdf->SetY($pdf->GetY() + DOC_INTERVAL * 2);
	
		// Affichage de l'image
		$pdf->Image('../tmp/report/'.$sdi_id.'_'.$current_scale.'.png', $_x, $_y, GRAPH_WIDTH * 0.30, GRAPH_HEIGHT * 0.30);
	
		// on place le curseur en bas du graphique
		$pdf->SetY($pdf->GetY() + (GRAPH_HEIGHT * 0.30));
	
	}
	
	if($indicator_type == 'qualitative') {

		$pdf->writeHTML(getQualitativeViz($values, $result_sdii[0], 'short', true, true), true, false, true, false, '');
		$pdf->Ln(5);
	}

	return true;
	
}

/**
 * indicator_table_values()
 * Affiche un tableau de valeurs a une échelle donnée
 *
 * @param int $sdi_id
 * @param int $current_scale
 * @param string $status
 * @param string $mode
 * @return bool true
 **/
function indicator_table_values($sdi_id, $current_scale, $status, $mode)
{
	
	global $pdf;
	global $sql_object;
	global $scale_denomination_array;

	$values ='';

	// setting $default_threshold value
	// By default initial value
	$result_sdii = $sql_object -> DBSelect(SQL_getInfoSdi($sdi_id));
	$default_threshold = $result_sdii[0]['sdii_threshold_value'];

	$sdi_values = $sql_object->DBSelect(SQL_getAllValue($status, $current_scale, $sdi_id));
	

	if (count($sdi_values) <= 1 && $sdi_values[0] == false) return false;

	// we prepare values for displaying
	for($i = 0; $i < count($sdi_values); $i++) {

		if(!empty($sdi_values[$i]['sdiv_multivalue'])) {

			$data = unserialize($sdi_values[$i]['sdiv_multivalue']);
			$current_value = '<p style="text-align:left;">';

			foreach($data as &$val) {
				$current_value .= '';
				if(!empty($val['label'])) $current_value .= $val['label']. ' : ';
				$current_value .= $val['value']. '<span class="value-percent"> ('.fnumber_format($val['_percentage'], 2, false).' %)</span>';
				$current_value .= '<br />';
			}
			// we display the total
			$current_value .= '<strong>'. mb_ucfirst(_t('dashboard', 'multivalue_total')) . ' : ' . $data[0]['_total']. '</strong>';
			$current_value .= '</p>';
			
			if($i == 0) {
				$lastValue = $data[0]['_total']; // we store data to compute TCAM and global rate
				$year_end = substr(formatDate($sdi_values[$i]['date_p'], true), 0, 4);
			}
			if($i == count($sdi_values)-1) {
				$firstValue = $data[0]['_total']; // we store data to compute TCAM and global rate
				$year_start = substr(formatDate($sdi_values[$i]['date_p'], true), 0, 4);
			}
		} else {
			
			$current_value = '';
			
			// If indicator is boolean or quantitative
			if($result_sdii[0]['sdii_type'] == 'boolean' || $result_sdii[0]['sdii_type'] == 'qualitative') {
				$a = getBooleanValues($result_sdii[0]);
				if($a) {
					$current_value .= $a[$sdi_values[$i]['sdiv_value']];
				} else {
					$current_value .= fnumber_format($sdi_values[$i]['sdiv_value'], 'auto', false);
				}
				 
			} else {
				if($i == 0) {
					$lastValue = $sdi_values[$i]['sdiv_value']; // we store data to compute TCAM and global rate
					$year_end = substr(formatDate($sdi_values[$i]['date_p'], true), 0, 4);
				}
				if($i == count($sdi_values)-1) {
					$firstValue = $sdi_values[$i]['sdiv_value']; // we store data to compute TCAM and global rate
					$year_start = substr(formatDate($sdi_values[$i]['date_p'], true), 0, 4);
				}
				$current_value .= fnumber_format($sdi_values[$i]['sdiv_value'], 'auto', false);
			}
			
		}
		 
		// if set, we get it from the table
		if(!is_null($sdi_values[$i]['sdiv_threshold'])) {
			// If indicator is boolean or quantitative
			if($result_sdii[0]['sdii_type'] == 'boolean' || $result_sdii[0]['sdii_type'] == 'qualitative') {
				$threshold = $a[$sdi_values[$i]['sdiv_threshold']];
			} else {
				$threshold = empty_none(fnumber_format($sdi_values[$i]['sdiv_threshold'], 'auto', false));
			}
			
		} else {
			// If indicator is boolean or quantitative
			if($result_sdii[0]['sdii_type'] == 'boolean' || $result_sdii[0]['sdii_type'] == 'qualitative') {
				if(!is_null($default_threshold)) $threshold = $a[$default_threshold];
				else $threshold = empty_none('');
			} else {
				$threshold = empty_none(fnumber_format($default_threshold, 'auto', false));
			}
		}
		 
		if ($sdi_values[$i]['sdiv_statut'] == 'D') $current_value .= ' *';
		if (!empty($sdi_values[$i]['sdiv_comment'])) {
			if($sdi_values[$i]['sdiv_comment_display']=='Y') {
				$comment_status = _t('dashboard', 'public');
			} else {
				$comment_status = _t('dashboard', 'private');
			}
			
			// $comments = formatText(strip_tags($sdi_values[$i]['sdiv_comment'])) .' ('.$comment_status.')<br />';
			$converted = new html2text($sdi_values[$i]['sdiv_comment']);
			$comments = formatText($converted->get_text())  .' ('.$comment_status.')<br />';

		} else {
			$comments = '';
		}
		
		$current_date = formatText($sdi_values[$i]['date_p']);
		$values .= '
		<tr nobr="true">
		<td class="brb">'.$current_value.'</td>
		<td class="brb">'.$threshold.'</td>
		<td class="brb">'.$current_date.'</td>
		<td class="brb">'.$comments.'</td>
		</tr>';
	}


	$tbl = '<style>
		table {
  			border:1px solid #E2E2E2;
  		}
	  .header {
	  		color:#777777;
				background-color:#E2E2E2;
	  		font-weight:bold;
	  		font-size:7pt;
	  	}
	  td {
	    	text-align:center;
	  		color:#777777;
	    }
	  td.brb {
	  		border-right:1px solid #E2E2E2;
				border-bottom:1px solid #E2E2E2;
	  }
    td.comment {
    }
	</style>
	<table cellspacing="0" cellpadding="5">
	<thead>
	<tr>
	<td class="header">'.mb_ucfirst(_t('dashboard', 'name_pluriel')).'</td>
	<td class="header">'.mb_ucfirst(_t('dashboard', 'threshold_value')).'</td>
	<td class="header">'.mb_ucfirst(_t('statut', 'published_on')).'</td>
	<td class="header">'.mb_ucfirst(_t('sdi', 'comment')).'</td>
	</tr>
	</thead>
	'.$values.'
	</table>';

	// Affichage des données

	// Affichage du titre seulement si affichage des valeurs en standalone
	// ou si la génération des graphiques est infaisables car moins de 3 valeurs
	if ($mode != 'TWICE' || (count($sdi_values) >= 1 && count($sdi_values) < 3)) {
		$out_scale_item = $scale_denomination_array[$current_scale];
		//$pdf->Bookmark($out_scale_item, 2, -1);
		indicator_format_item($out_scale_item);
		$pdf->SetY($pdf->GetY() + DOC_INTERVAL * 2);
	}
	
	$pdf->SetFont(DOC_POLICE, DOC_STYLE, ARRAY_DATA_SIZE );
	// Affichage du tableau
	// Affichage de l'astérisque - avertissement
	$pdf->MultiCell(0, DOC_HEIGHT, '* ' . strtolower(_t('statut', 'draftpdf')), 0, 'R', 0);
	$pdf->writeHTML($tbl, true, false, false, false, '');
	// affichage des taux
	if($result_sdii[0]['sdii_type'] == 'quantitative') {
		$rates = getRates($firstValue, $lastValue, $year_start, $year_end, 'txt');
		$pdf->MultiCell(0, DOC_HEIGHT, $rates, 0, 'L', 0);
	}	
	
	$pdf->SetFont(DOC_POLICE, DOC_STYLE, DOC_SIZE);
	$pdf->Ln(DOC_INTERVAL);
		
	return true;
}

?>