<?php
/**
 * Search module
 * 
 * Provides basics methods called from public templates
 * These methods can be overriden. Discover logic and examples on :
 * @url http://dev.linea21.com/projects/linea21/wiki/Override
 * @url http://dev.linea21.com/projects/linea21/wiki/Personnalisation_de_l%27interface_publique
 * 
 * @package linea21\core\search
 * @author $Author$ - linea21 <info@linea21.com>
 * @version $Id$ 
 * @access public
 * @license http://opensource.org/licenses/gpl-3.0.html
 */

include_once(override('../search/' . SQL . '.inc.php'));

/**
 * MatchDataFormat()
 *
 * @param string $keywords
 * @param string $searchtype
 * @return string $formatted_keywords
 */
if(!function_exists('MatchDataFormat')) {

  function MatchDataFormat($keywords, $searchtype)
  {
  	
    $keywords = trim(addslashes($keywords));
    // https://stackoverflow.com/questions/26507087/escape-string-for-use-in-mysql-fulltext-search
    // https://dev.linea21.com/issues/912
    $keywords = preg_replace('/[^\p{L}\p{N}_]+/u', ' ', $keywords);
    $keywords = preg_replace('/[+\-><\(\)~*\"@]+/', ' ', $keywords);
    
    // if MODE_SEARCH is equal to 'WITH QUERY EXPANSION' we do not format keywords
    // or $searchtype === 'expanded'
    if((defined('SEARCH_MODE') && stripos(SEARCH_MODE, 'boolean') === false) || $searchtype === 'expanded') return $keywords;
    
    switch ($searchtype) {
      case 'one':
        if(strtolower(SQL)=='pgsql') {
          $words = explode(' ', $keywords);
          $formatted_keywords = '';
          $op='';
          foreach($words as $value) {
            $formatted_keywords .= $op . $value;
            $op=' | ';
          }
          trim($formatted_keywords);
        }
        if(strtolower(SQL)=='mysql') {
          $formatted_keywords = $keywords;
        }
        break;
      case 'all':
        $words = explode(' ', $keywords);
        $formatted_keywords = '';
        $op='';
        foreach($words as $value) {
          $formatted_keywords .= $op . $value;
          if(strtolower(SQL)=='pgsql') $op=' & ';
          if(strtolower(SQL)=='mysql') $op=' + ';
        }
        trim($formatted_keywords);
        break;
      case 'exp':
        if(strtolower(SQL)=='pgsql') {
          $words = explode(' ', $keywords);
          $formatted_keywords = '';
          $op='';
          foreach($words as $value) {
            $formatted_keywords .= $op . $value;
            $op=' & ';
          }
          trim($formatted_keywords);
        }
        if(strtolower(SQL)=='mysql') {
          $formatted_keywords = '"' . $keywords . '"';
        }
        break;
      default:
        if(strtolower(SQL)=='pgsql') {
          $words = explode(' ', $keywords);
          $formatted_keywords = '';
          $op='';
          foreach($words as $value) {
            $formatted_keywords .= $op . $value;
            $op=' | ';
          }
          trim($formatted_keywords);
        }
        if(strtolower(SQL)=='mysql') {
          $formatted_keywords = $keywords;
        }
        break;
    }
    return $formatted_keywords;
  }
}

if(!function_exists('GetStatus')) {

  function GetStatus($short_status)
  {
    $write = ' - ';
    switch ($short_status) {
      case 'PA':
        $write .= _t('statut','PA');
        break;
      case 'AA':
        $write .= _t('statut','AA');
        break;
      case 'P':
        $write = '';
        break;
      case 'Y':
        $write = '';
        break;
      case 'D':
        $write .= _t('statut','draft');
        break;
      default:
        $write = '';
    }
    return $write;
  }
}

/**
 * DisplayPublicResults()
 *
 * @param string $keywords
 * @param string $searchtype
 * @return void (echo)
 **/
if(!function_exists('DisplayPublicResults')) {

  function DisplayPublicResults($keywords, $searchtype)
  {
    $type = 'PUBLIC';
    if(defined('MOD_PROJECT') && MOD_PROJECT == true) {
        DisplaySearchProject($keywords, $type, $searchtype);
        // if(defined('PROJECT_TASK') && PROJECT_TASK == 1) DisplaySearchProjectTasks($keywords, $type, $searchtype);
    }
    if(defined('MOD_NEWS') && MOD_NEWS == true) DisplaySearchNews($keywords, $type, $searchtype);
    //echo '<hr />';
    if(defined('MOD_PUBLICATION') && MOD_PUBLICATION == true) DisplaySearchPublication($keywords, $type, $searchtype);
    // echo '<hr />';
    if(defined('MOD_PUBLICATION') && MOD_PUBLICATION == true) DisplaySearchPubliContents($keywords, $type, $searchtype);
    //echo '<hr />';
    if(defined('MOD_WORKSHOP') && MOD_WORKSHOP == true) DisplaySearchWorkshop($keywords, $type, $searchtype);
    // echo '<hr />';
    if(defined('MOD_WORKSHOP') && MOD_WORKSHOP == true) DisplaySearchWorkshopRep($keywords, $type, $searchtype);
    // echo '<hr />';
    if(defined('MOD_WORKSHOP') && MOD_WORKSHOP == true) DisplaySearchWorkshopRepContents($keywords, $type, $searchtype);
    // echo '<hr />';
    if(defined('MOD_WORKSHOP') && MOD_WORKSHOP == true) DisplaySearchWorkshopCalendar($keywords, $type, $searchtype);
    // echo '<hr />';
    if(defined('MOD_WORKSHOP') && MOD_WORKSHOP == true) DisplaySearchWorkshopCom($keywords, $type, $searchtype);
    //echo '<hr />';
    if(defined('MOD_SDI') && MOD_SDI == true) DisplaySearchSdi($keywords, $type, $searchtype);
    //echo '<hr />';
    if(defined('MOD_YELLOWPAGES') && MOD_YELLOWPAGES == true) DisplaySearchYellowpages($keywords, $type, $searchtype);
  }
}

/**
 * DisplaySearchComment()
 * Affiche les projets répondant à la requête sur $keywords
 *
 * @param string $keywords
 * @return string $content
 */
if(!function_exists('DisplaySearchComment')) {

  function DisplaySearchComment($keywords, $type, $searchtype)
  {
    $data = $GLOBALS['sql_object']->DBSelect(SQL_comment_search(MatchDataFormat($keywords, $searchtype), $type, $searchtype));

    // no search on comments from public app
    if($type == 'PUBLIC') return true;

    $content = '<div id="MatchComment">';
    $content .= '<h3><i class="fa fa-comments-o" aria-hidden="true"></i> ' . _t('search','comment_module') . '</h3>' . PHP_EOL;
    if (!is_array($data)) $content .= '<p>' . _t('search','no_result') . '</p>' . PHP_EOL;
    else {
      $content .= '<ul>' . PHP_EOL;
      for($i = 0; $i < count($data); $i++) {
        if ($type == 'PUBLIC') {
          $link = HrefMaker(array('rub' => $GLOBALS['links'][U_L]['project']['linkvalue'], 'id' => $data[$i]['project_id'], 'name' => $data[$i]['project_name']));
          $published_display = _t('statut','published_on') . ' ' . $data[$i]['date_display'];
        }
        if ($type == 'ADMIN') {
          $link = 'index.php?rub=comment&amp;todo=det&amp;id=' . $data[$i]['comment_id'];
          $published_display = _t('statut','published_on') . ' ' . $data[$i]['date_display'];
        }
        if(!empty($data[$i]['comment_name'])) $uname = formatText($data[$i]['comment_name'], '2HTML');
        else $uname = formatText($data[$i]['user_login'], '2HTML');

        $title = $data[$i]['comment_body'];
        $pertinence = FormatPertinenceIndice($data[$i]['pertinence']);

        $content .= '<li class="matches">' . PHP_EOL;
        $content .= '<a href="' . $link . '" title="' . formatText(cutText($title, 300), '2ATT') . '" >' . formatText(cutText($title, 200), '2HTML') . '</a>';
        $content .= '<span class="infocomp"> - ' . mb_ucfirst($published_display). ' ' . _t('divers', 'by'). ' ' . $uname . ' ' . GetStatus($data[$i]['comment_status']) . '</span>';
        $content .= $pertinence;
        $content .= '</li>' . PHP_EOL;
      }
      $content .= '</ul>' . PHP_EOL;
    }
    $content .= '</div>' . PHP_EOL;
    echo $content;
  }
}

/**
 * DisplaySearchProjectTasks()
 * Affiche les projets répondant à la requête sur $keywords
 *
 * @param string $keywords
 * @return string $content
 */
if(!function_exists('DisplaySearchProjectTasks')) {
    
    function DisplaySearchProjectTasks($keywords, $type, $searchtype)
    {
        $data = $GLOBALS['sql_object']->DBSelect(SQL_project_task_search(MatchDataFormat($keywords, $searchtype), $type, $searchtype));

        $content = '<div id="MatchProjectTask">';
        $content .= '<h3><i class="fa fa-briefcase" aria-hidden="true"></i> ' . _t('search','project_task_module') . '</h3>' . PHP_EOL;
        if (!is_array($data) ) $content .= '<p>' . _t('search','no_result') . '</p>' . PHP_EOL;
        else {
            $collected = array();
            $content .= '<ul>' . PHP_EOL;
            for($i = 0; $i < count($data); $i++) {
                
                if(!in_array($data[$i]['project_id'], $collected)) {
                    if ($type == 'PUBLIC') {
                        $link = HrefMaker(array('rub' => $GLOBALS['links'][U_L]['project']['linkvalue'], 'id' => $data[$i]['project_id'], 'name' => $data[$i]['project_name']));
                        $published_display = _t('statut','published_on') . ' ' . $data[$i]['date_display'];
                    }
                    if ($type == 'ADMIN') {
                        $link = 'index.php?rub=project&amp;todo=det&amp;id=' . $data[$i]['project_id'].'#task-'.$data[$i]['task_id'];
                        $published_display = _t('divers','date_crea') . ' : ' . $data[$i]['date_display'];
                    }
                    
                    $news_title = formatText($data[$i]['project_name'], '2HTML');
                    $pertinence = FormatPertinenceIndice($data[$i]['pertinence']);
                    
                    $content .= '<li class="matches">' . PHP_EOL;
                    $content .= '<a href="' . $link . '" title="' . formatText($GLOBALS['links'][U_L]['project']['desc'] . ' : ' . $news_title, '2ATT') . '" >' . $news_title . '</a>';
                    $content .= '<span class="infocomp"> - ' .mb_ucfirst($published_display) . GetStatus($data[$i]['statut']) . '</span>';
                    $content .= $pertinence;
                    $content .= '</li>' . PHP_EOL;
                    
                    array_push($collected, $data[$i]['project_id']);
                }
            }
            $content .= '</ul>' . PHP_EOL;
        }
        $content .= '</div>' . PHP_EOL;
        echo $content;
    }
}

/**
 * DisplaySearchProject()
 * Affiche les projets répondant à la requête sur $keywords
 *
 * @param string $keywords
 * @return string $content
 */
if(!function_exists('DisplaySearchProject')) {

  function DisplaySearchProject($keywords, $type, $searchtype)
  {
    $data = $GLOBALS['sql_object']->DBSelect(SQL_project_search(MatchDataFormat($keywords, $searchtype), $type, $searchtype));

    $content = '<div id="MatchProject">';
    $content .= '<h3><i class="fa fa-briefcase" aria-hidden="true"></i> ' . _t('search','project_module') . '</h3>' . PHP_EOL;
    if (!is_array($data) ) $content .= '<p>' . _t('search','no_result') . '</p>' . PHP_EOL;
    else {
      $content .= '<ul>' . PHP_EOL;
      for($i = 0; $i < count($data); $i++) {
        if ($type == 'PUBLIC') {
          $link_news = HrefMaker(array('rub' => $GLOBALS['links'][U_L]['project']['linkvalue'], 'id' => $data[$i]['project_id'], 'name' => $data[$i]['project_name']));
          $published_display = _t('statut','published_on') . ' ' . $data[$i]['date_display'];
        }
        if ($type == 'ADMIN') {
          $link_news = 'index.php?rub=project&amp;todo=det&amp;id=' . $data[$i]['project_id'];
          $published_display = _t('divers','date_crea') . ' : ' . $data[$i]['date_display'];
        }

        $news_title = formatText($data[$i]['project_name'], '2HTML');
        $pertinence = FormatPertinenceIndice($data[$i]['pertinence']);

        $content .= '<li class="matches">' . PHP_EOL;
        $content .= '<a href="' . $link_news . '" title="' . formatText($GLOBALS['links'][U_L]['project']['desc'] . ' : ' . $news_title, '2ATT') . '" >' . $news_title . '</a>';
        $content .= '<span class="infocomp"> - ' .mb_ucfirst($published_display) . GetStatus($data[$i]['statut']) . '</span>';
        $content .= $pertinence;
        $content .= '</li>' . PHP_EOL;
      }
      $content .= '</ul>' . PHP_EOL;
    }
    $content .= '</div>' . PHP_EOL;
    echo $content;
  }
}

/**
 * DisplaySearchPage()
 * Affiche les pages répondant à la requête sur $keywords
 *
 * @param string $keywords
 * @return string $content
 */
if(!function_exists('DisplaySearchPage')) {

	function DisplaySearchPage($keywords, $type, $searchtype)
	{
		$data = $GLOBALS['sql_object']->DBSelect(SQL_page_search(MatchDataFormat($keywords, $searchtype), $type, $searchtype));

		$content = '<div id="MatchNews">';
		$content .= '<h3><i class="fa fa-file-o" aria-hidden="true"></i> ' . _t('search','page_module') . '</h3>' . PHP_EOL;
		if (!is_array($data)) $content .= '<p>' . _t('search','no_result') . '</p>' . PHP_EOL;
		else {
			$content .= '<ul>' . PHP_EOL;
			for($i = 0; $i < count($data); $i++) {
// @todo see if necessary to implement or not				
// 				if ($type == 'PUBLIC') {
// 					$link = HrefMaker(array('rub' => $GLOBALS['links'][U_L]['page']['linkvalue'], 'id' => $data[$i]['news_id'], 'name' => $data[$i]['news_title']));
// 					$published_display = _t('statut','published_on') . ' ' . $data[$i]['date_display'];
// 				}
				if ($type == 'ADMIN') {
					$link = 'index.php?rub=page&amp;todo=det&amp;id=' . $data[$i]['page_id'];
					$published_display = _t('divers','date_crea') . ' : ' . $data[$i]['date_display'];
				}

				$title = formatText($data[$i]['page_title'], '2HTML');
				$pertinence = FormatPertinenceIndice($data[$i]['pertinence']);

				$content .= '<li class="matches">' . PHP_EOL;
				$content .= '<a href="' . $link . '" title="' . formatText($GLOBALS['links'][U_L]['news']['desc'] . ' : ' . $title, '2ATT') . '" >' . $title . '</a>';
				$content .= '<span class="infocomp"> - ' . mb_ucfirst($published_display) . GetStatus($data[$i]['statut']) . '</span>';
				$content .= $pertinence;
				$content .= '</li>' . PHP_EOL;
			}
			$content .= '</ul>' . PHP_EOL;
		}
		$content .= '</div>' . PHP_EOL;
		echo $content;
	}
}

/**
 * DisplaySearchNews()
 * Affiche les news répondant à la requête sur $keywords
 *
 * @param string $keywords
 * @return string $content
 */
if(!function_exists('DisplaySearchNews')) {

  function DisplaySearchNews($keywords, $type, $searchtype)
  {
    $data = $GLOBALS['sql_object']->DBSelect(SQL_news_search(MatchDataFormat($keywords, $searchtype), $type, $searchtype));

    $content = '<div id="MatchNews">';
    $content .= '<h3><i class="fa fa-calendar" aria-hidden="true"></i> ' . _t('search','news_module') . '</h3>' . PHP_EOL;
    if (!is_array($data)) $content .= '<p>' . _t('search','no_result') . '</p>' . PHP_EOL;
    else {
      $content .= '<ul>' . PHP_EOL;
      for($i = 0; $i < count($data); $i++) {
        if ($type == 'PUBLIC') {
          $link_news = HrefMaker(array('rub' => $GLOBALS['links'][U_L]['news']['linkvalue'], 'id' => $data[$i]['news_id'], 'name' => $data[$i]['news_title']));
          $published_display = _t('statut','published_on') . ' ' . $data[$i]['date_display'];
        }
        if ($type == 'ADMIN') {
          $link_news = 'index.php?rub=news&amp;todo=det&amp;id=' . $data[$i]['news_id'];
          $published_display = _t('divers','date_crea') . ' : ' . $data[$i]['date_display'];
        }

        $news_title = formatText($data[$i]['news_title'], '2HTML');
        $pertinence = FormatPertinenceIndice($data[$i]['pertinence']);

        $content .= '<li class="matches">' . PHP_EOL;
        $content .= '<a href="' . $link_news . '" title="' . formatText($GLOBALS['links'][U_L]['news']['desc'] . ' : ' . $news_title, '2ATT') . '" >' . $news_title . '</a>';
        $content .= '<span class="infocomp"> - ' . mb_ucfirst($published_display) . GetStatus($data[$i]['statut']) . '</span>';
        $content .= $pertinence;
        $content .= '</li>' . PHP_EOL;
      }
      $content .= '</ul>' . PHP_EOL;
    }
    $content .= '</div>' . PHP_EOL;
    echo $content;
  }
}

/**
 * DisplaySearchNews()
 * Affiche les news répondant à la requête sur $keywords
 *
 * @param string $keywords
 * @return string $content
 */
if(!function_exists('DisplaySearchNewsletter')) {

  function DisplaySearchNewsletter($keywords, $type, $searchtype)
  {
    $data = $GLOBALS['sql_object']->DBSelect(SQL_newsletter_search(MatchDataFormat($keywords, $searchtype), $type, $searchtype));

    $content = '<div id="MatchNewsletter">';
    $content .= '<h3><i class="fa fa-paper-plane" aria-hidden="true"></i> ' . _t('search','newsletter_module') . '</h3>' . PHP_EOL;
    if (!is_array($data)) $content .= '<p>' . _t('search','no_result') . '</p>' . PHP_EOL;
    else {
      $content .= '<ol>' . PHP_EOL;
      for($i = 0; $i < count($data); $i++) {
        /**
         * * NON IMPLEMENTE
         * if ($type == 'PUBLIC') {
         * $link_news = HrefMaker(array('rub' => $GLOBALS['links'][U_L]['news']['linkvalue'], 'id' => $data[$i]['news_id']));
         * $published_display = _t('statut','published_on') . ' ' . $data[$i]['date_display'];
         * }
         */

        if ($type == 'ADMIN') {
          $link_newsletter = 'index.php?rub=newsletter&amp;todo=det&amp;id=' . $data[$i]['newsletter_id'];
          $published_display = _t('divers','date_crea') . ' : ' . $data[$i]['date_display'];
        }
        $newsletter_title = formatText($data[$i]['newsletter_title'], '2HTML');
        $pertinence = FormatPertinenceIndice($data[$i]['pertinence']);

        $content .= '<li class="matches">' . PHP_EOL;
        $content .= '<a href="' . $link_newsletter . '" title="' . formatText($newsletter_title, '2ATT') . '" >' . $newsletter_title . '</a>';
        $content .= '<span class="infocomp"> - ' . mb_ucfirst($published_display) . GetStatus($data[$i]['statut']) . '</span>';
        $content .= $pertinence;
        $content .= '</li>' . PHP_EOL;
      }
      $content .= '</ol>' . PHP_EOL;
    }
    $content .= '</div>' . PHP_EOL;
    echo $content;
  }
}

/**
 * DisplaySearchPublication()
 * Affiche les publications répondant à la requête sur $keywords
 * Requête éffectuée sur le titre et le résumé des publications
 *
 * @param string $keywords
 * @return string $content
 */
if(!function_exists('DisplaySearchPublication')) {

  function DisplaySearchPublication($keywords, $type, $searchtype)
  {
    $data = $GLOBALS['sql_object']->DBSelect(SQL_publication_search(MatchDataFormat($keywords, $searchtype), $type, $searchtype));

    $content = '<div id="MatchPubli">';
    $content .= '<h3><i class="fa fa-book" aria-hidden="true"></i> ' . _t('search','publication_module') . '</h3>' . PHP_EOL;
    if (!is_array($data)) $content .= '<p>' . _t('search','no_result') . '</p>' . PHP_EOL;
    else {
      $content .= '<ul>' . PHP_EOL;
      for($i = 0; $i < count($data); $i++) {
        if ($type == 'PUBLIC') {
          $link_publi = HrefMaker(array('rub' => $GLOBALS['links'][U_L]['publication']['linkvalue'], 'id' => $data[$i]['publi_id'], 'name' => $data[$i]['publi_title']));
          $published_display = _t('statut','published_on') . ' ' . $data[$i]['date_display'];
        }
        if ($type == 'ADMIN') {
          $link_publi = 'index.php?rub=publication&amp;todo=det&amp;id=' . $data[$i]['publi_id'];
          $published_display = _t('divers','date_crea') . ' : ' . $data[$i]['date_display'];
        }

        $publi_title = formatText($data[$i]['publi_title'], '2HTML');
        $pertinence = FormatPertinenceIndice($data[$i]['pertinence']);

        $content .= '<li class="matches">' . PHP_EOL;
        $content .= '<a href="' . $link_publi . '" title="' . formatText($GLOBALS['links'][U_L]['publication']['desc'] . ' : ' . $publi_title, '2ATT') . '" >' . $publi_title . '</a>';
        $content .= '<span class="infocomp"> - ' . mb_ucfirst($published_display) . GetStatus($data[$i]['statut']) . '</span>';
        $content .= $pertinence;
        $content .= '</li>' . PHP_EOL;
      }
      $content .= '</ul>' . PHP_EOL;
    }
    $content .= '</div>' . PHP_EOL;
    echo $content;
  }
}

/**
 * DisplaySearchPubliContents()
 * Affiche les publications répondant à la requête sur $keywords
 * Requête éffectuée sur le contenu des publications ( titre & body )
 *
 * @param string $keywords
 * @return string $content
 */
if(!function_exists('DisplaySearchPubliContents')) {

  function DisplaySearchPubliContents($keywords, $type, $searchtype)
  {
    $data = $GLOBALS['sql_object']->DBSelect(SQL_publication_content_search(MatchDataFormat($keywords, $searchtype), $type, $searchtype));

    $content = '<div id="MatchPubliContents">';
    $content .= '<h4>' . _t('search','det_publication_module') . '</h4>' . PHP_EOL;
    //if (!is_array($data)) $content .= '<p>' . _t('search','no_result') . '</p>' . PHP_EOL;
    if (!is_array($data)) return true;
    else {
      $content .= '<ul>' . PHP_EOL;
      for($i = 0; $i < count($data); $i++) {
        if ($type == 'PUBLIC') {
          $link_publi_det = HrefMaker(array('rub' => $GLOBALS['links'][U_L]['publication-part']['linkvalue'], 'id' => $data[$i]['publicon_id'], 'parentid' => $data[$i]['publi_id'], 'name' => $data[$i]['publicon_title']));
          $link_publi = HrefMaker(array('rub' => $GLOBALS['links'][U_L]['publication']['linkvalue'], 'id' => $data[$i]['publi_id'], 'name' => $data[$i]['publi_title']));
          $published_display = _t('statut','published_on') . ' ' . $data[$i]['date_display'];
        }
        if ($type == 'ADMIN') {
          $link_publi_det = 'index.php?rub=contents&amp;todo=det&amp;parent=' . $data[$i]['publi_id'] . '&amp;type=P&amp;id=' . $data[$i]['publicon_id'];
          $link_publi = 'index.php?rub=publication&amp;todo=det&amp;id=' . $data[$i]['publi_id'];
          $published_display = _t('divers','date_crea') . ' : ' . $data[$i]['date_display'];
        }

        $publicon_title = formatText($data[$i]['publicon_title'], '2HTML');
        $publi_title = formatText($data[$i]['publi_title'], '2HTML');
        $pertinence = FormatPertinenceIndice($data[$i]['pertinence']);

        $content .= '<li class="matches">' . PHP_EOL;
        $content .= '<a href="' . $link_publi_det . '" title="' . formatText($GLOBALS['links'][U_L]['publication-part']['desc'] . ' : ' . $publicon_title, '2ATT') . '" >' . $publicon_title . '</a>';
        $content .= '<span class="infocomp"> - ' .  mb_ucfirst($published_display) . '</span>';
        $content .= '<br />';
        $content .= '<span class="infoparent">' . mb_ucfirst(_t('name','publication_module')) . ' : ';
        $content .= '<a href="' . $link_publi . '" title="' . formatText($GLOBALS['links'][U_L]['publication']['desc'] . ' : ' . $publi_title, '2ATT') . '" >' . $publi_title . '</a>';
        $content .= '</span>';
        $content .= $pertinence;
        $content .= '</li>' . PHP_EOL;
      }
      $content .= '</ul>' . PHP_EOL;
    }
    $content .= '</div>' . PHP_EOL;
    echo $content;
  }
}

/**
 * DisplaySearchWorkshop()
 * Affiche les groupes de travail répondant à la requête sur $keywords
 * Requête éffectuée sur le libellé et le résumé des groupes de travail
 *
 * @param string $keywords
 * @return string $content
 */
if(!function_exists('DisplaySearchWorkshop')) {

  function DisplaySearchWorkshop($keywords, $type, $searchtype)
  {
    $data = $GLOBALS['sql_object']->DBSelect(SQL_workshop_search(MatchDataFormat($keywords, $searchtype), $type, $searchtype));

    $content = '<div id="MatchWorkshop">';
    $content .= '<h3><i class="fa fa-users" aria-hidden="true"></i> ' . _t('search','workshop_module') . '</h3>' . PHP_EOL;
    if (!is_array($data)) $content .= '<p>' . _t('search','no_result') . '</p>' . PHP_EOL;
    else {
      $content .= '<ul>' . PHP_EOL;
      for($i = 0; $i < count($data); $i++) {
        if ($type == 'PUBLIC') {
          $link_workshop = HrefMaker(array('rub' => $GLOBALS['links'][U_L]['workgroup']['linkvalue'], 'id' => $data[$i]['workshop_id'], 'name' => $data[$i]['workshop_denomination']));
        }
        if ($type == 'ADMIN') {
          $link_workshop = 'index.php?rub=workshop&amp;todo=det&amp;id=' . $data[$i]['workshop_id'];
        }

        $workshop_title = formatText($data[$i]['workshop_denomination'], '2HTML');
        $pertinence = FormatPertinenceIndice($data[$i]['pertinence']);
        $published_display = _t('divers','date_crea') . ' : ' . $data[$i]['date_display'];
        
        $content .= '<li class="matches">' . PHP_EOL;
        $content .= '<a href="' . $link_workshop . '" title="' . formatText($GLOBALS['links'][U_L]['workgroup']['desc'] . ' : ' . $workshop_title, '2ATT') . '" >' . $workshop_title . '</a>';
        $content .= '<span class="infocomp"> - ' . mb_ucfirst($published_display) . GetStatus($data[$i]['statut']) . '</span>';
        $content .= $pertinence;
        $content .= '</li>' . PHP_EOL;
      }
      $content .= '</ul>' . PHP_EOL;
    }
    $content .= '</div>' . PHP_EOL;
    echo $content;
  }
}

/**
 * DisplaySearchWorkshopRep()
 * Affiche les groupes de travail répondant à la requête sur $keywords
 * Requête éffectuée sur le contenu des rapports de groupes de travail ( titre & body )
 *
 * @param string $keywords
 * @return string $content
 */
if(!function_exists('DisplaySearchWorkshopRep')) {

  function DisplaySearchWorkshopRep($keywords, $type, $searchtype)
  {
    $data = $GLOBALS['sql_object']->DBSelect(SQL_workrep_search(MatchDataFormat($keywords, $searchtype), $type, $searchtype));

    $content = '<div id="MatchWorkshopRep">';
    $content .= '<h4>' . _t('search','workshoprep_module') . '</h4>' . PHP_EOL;
    //if (!is_array($data)) $content .= '<p>' . _t('search','no_result') . '</p>' . PHP_EOL;
    if (!is_array($data)) return true;
    else {
      $content .= '<ul>' . PHP_EOL;
      for($i = 0; $i < count($data); $i++) {
        if ($type == 'PUBLIC') {
          $link_workrep = HrefMaker(array('rub' => $GLOBALS['links'][U_L]['report']['linkvalue'], 'id' => $data[$i]['workrep_id'], 'name' => $data[$i]['workrep_title']));
          $link_workshop = HrefMaker(array('rub' => $GLOBALS['links'][U_L]['workgroup']['linkvalue'], 'id' => $data[$i]['workshop_id'], 'name' => $data[$i]['workshop_denomination']));
          $published_display = _t('statut','published_on') . ' ' . $data[$i]['date_display'];
        }
        if ($type == 'ADMIN') {
          $link_workrep = 'index.php?rub=workshoprep&amp;todo=det&amp;id=' . $data[$i]['workrep_id'];
          $link_workshop = 'index.php?rub=workshop&amp;todo=det&amp;id=' . $data[$i]['workshop_id'];
          $published_display = _t('divers','date_crea') . ' : ' . $data[$i]['date_display'];
        }

        $workrep_title = formatText($data[$i]['workrep_title'], '2HTML');
        $workshop_title = formatText($data[$i]['workshop_denomination'], '2HTML');
        $pertinence = FormatPertinenceIndice($data[$i]['pertinence']);

        $content .= '<li class="matches">' . PHP_EOL;
        $content .= '<a href="' . $link_workrep . '" title="' . formatText($GLOBALS['links'][U_L]['report']['desc'] . ' : ' . $workrep_title, '2ATT') . '" >' . $workrep_title . '</a>';
        $content .= '<span class="infocomp"> - ' . mb_ucfirst($published_display) . GetStatus($data[$i]['statut']) . '</span>';
        $content .= '<br />';
        $content .= '<span class="infoparent">' . mb_ucfirst(_t('name','workshop_module')) . ' : ';
        $content .= '<a href="' . $link_workshop . '" title="' . formatText($GLOBALS['links'][U_L]['workgroup']['desc'] . ' : ' . $workshop_title, '2ATT') . '" >' . $workshop_title . '</a>';
        $content .= '</span>';
        $content .= $pertinence;
        $content .= '</li>' . PHP_EOL;
      }
      $content .= '</ul>' . PHP_EOL;
    }
    $content .= '</div>' . PHP_EOL;
    echo $content;
  }
}

/**
 * DisplaySearchWorkshopRepContents()
 * Affiche les groupes de travail répondant à la requête sur $keywords
 * Requête éffectuée sur le contenu des rapports de groupes de travail ( titre & body )
 *
 * @param string $keywords
 * @return string $content
 */
if(!function_exists('DisplaySearchWorkshopRepContents')) {

  function DisplaySearchWorkshopRepContents($keywords, $type, $searchtype)
  {
    $data = $GLOBALS['sql_object']->DBSelect(SQL_workrep_content_search(MatchDataFormat($keywords, $searchtype), $type, $searchtype));

    $content = '<div id="MatchWorkshopContents">';
    $content .= '<h5>' . _t('search','workshoprepcon_module') . '</h5>' . PHP_EOL;
    //if (!is_array($data)) $content .= '<p>' . _t('search','no_result') . '</p>' . PHP_EOL;
    if (!is_array($data)) return true;
    else {
      $content .= '<ul>' . PHP_EOL;
      for($i = 0; $i < count($data); $i++) {
        if ($type == 'PUBLIC') {
          $link_workrep_det = HrefMaker(array('rub' => $GLOBALS['links'][U_L]['report-part']['linkvalue'], 'id' => $data[$i]['workrepcon_id'], 'parentid' => $data[$i]['workrep_id'], 'name' => $data[$i]['workrepcon_title']));
          $link_workrep = HrefMaker(array('rub' => $GLOBALS['links'][U_L]['report']['linkvalue'], 'id' => $data[$i]['workrep_id'], 'name' => $data[$i]['workrep_title']));
          $published_display = _t('statut','published_on') . ' ' . $data[$i]['date_display'];
        }
        if ($type == 'ADMIN') {
          $link_workrep_det = 'index.php?rub=contents&amp;todo=det&amp;parent=' . $data[$i]['workrep_id'] . '&amp;type=W&amp;id=' . $data[$i]['workrepcon_id'];
          $link_workrep = 'index.php?rub=workshoprep&amp;todo=det&amp;id=' . $data[$i]['workrep_id'];
          $published_display = _t('divers','date_crea') . ' : ' . $data[$i]['date_display'];
        }

        $workrepcon_title = formatText($data[$i]['workrepcon_title'], '2HTML');
        $workrep_title = formatText($data[$i]['workrep_title'], '2HTML');
        $pertinence = FormatPertinenceIndice($data[$i]['pertinence']);

        $content .= '<li class="matches">' . PHP_EOL;
        $content .= '<a href="' . $link_workrep_det . '" title="' . formatText($GLOBALS['links'][U_L]['publication-part']['desc'] . ' : ' . $workrepcon_title, '2ATT') . '" >' . $workrepcon_title . '</a>';
        $content .= '<span class="infocomp"> - ' . mb_ucfirst($published_display) . GetStatus($data[$i]['statut']) . '</span>';
        $content .= '<br />';
        $content .= '<span class="infoparent">' . mb_ucfirst(_t('name','workshoprep_module')) . ' : ';
        $content .= '<a href="' . $link_workrep . '" title="' . formatText($GLOBALS['links'][U_L]['report']['desc'] . ' : ' . $workrep_title, '2ATT') . '" >' . $workrep_title . '</a>';
        $content .= '</span>';
        $content .= $pertinence;
        $content .= '</li>' . PHP_EOL;
      }
      $content .= '</ul>' . PHP_EOL;
    }
    $content .= '</div>' . PHP_EOL;
    echo $content;
  }
}

/**
 * DisplaySearchWorkshopCalendar()
 * Affiche les groupes de travail répondant à la requête sur $keywords
 * Requête éffectuée sur les tâches du calendrier de groupes de travail ( libellé & contenu des tâches )
 *
 * @param string $keywords
 * @return string $content
 */
if(!function_exists('DisplaySearchWorkshopCalendar')) {

  function DisplaySearchWorkshopCalendar($keywords, $type, $searchtype)
  {
    $data = $GLOBALS['sql_object']->DBSelect(SQL_workshop_calendar_search(MatchDataFormat($keywords, $searchtype), $type, $searchtype));

    $content = '<div id="MatchWorkshopCal">';
    $content .= '<h4>' . _t('search','workshopcal_module') . '</h4>' . PHP_EOL;
    //if (!is_array($data)) $content .= '<p>' . _t('search','no_result') . '</p>' . PHP_EOL;
    if (!is_array($data)) return true;
    else {
      $content .= '<ul>' . PHP_EOL;
      for($i = 0; $i < count($data); $i++) {
        if ($type == 'PUBLIC') {
          $link_workcal = HrefMaker(array('rub' => $GLOBALS['links'][U_L]['calendar']['linkvalue'], 'id' => $data[$i]['workcal_workshop_id'], 'name' => $data[$i]['workcal_task']));
          $link_workshop = HrefMaker(array('rub' => $GLOBALS['links'][U_L]['workgroup']['linkvalue'], 'id' => $data[$i]['workcal_workshop_id'], 'name' => $data[$i]['workshop_denomination']));
        }
        if ($type == 'ADMIN') {
          $link_workcal = 'index.php?rub=workshop&amp;todo=list_cal&amp;id=' . $data[$i]['workcal_workshop_id'];
          $link_workshop = 'index.php?rub=workshop&amp;todo=det&amp;id=' . $data[$i]['workcal_workshop_id'];
        }

        $workcal_title = formatText($data[$i]['workcal_task'], '2HTML');
        $workshop_title = formatText($data[$i]['workshop_denomination'], '2HTML');
        $pertinence = FormatPertinenceIndice($data[$i]['pertinence']);
        $published_display = _t('statut','published_on') . ' ' . $data[$i]['date_display'];

        $content .= '<li class="matches">' . PHP_EOL;
        $content .= '<a href="' . $link_workcal . '" title="' . formatText($GLOBALS['links'][U_L]['calendar']['desc'] . ' : ' . $workcal_title, '2ATT') . '" >' . $workcal_title . '</a>';
        $content .= '<span class="infocomp"> - ' . mb_ucfirst($published_display) . '</span>';
        $content .= '<br />';
        $content .= '<span class="infoparent">' . mb_ucfirst(_t('name','workshop_module')). ' : ';
        $content .= '<a href="' . $link_workshop . '" title="' . formatText($GLOBALS['links'][U_L]['workgroup']['desc'] . ' : ' . $workshop_title, '2ATT') . '" >' . $workshop_title . '</a>';
        $content .= '</span>';
        $content .= $pertinence;
        $content .= '</li>' . PHP_EOL;
      }
      $content .= '</ul>' . PHP_EOL;
    }
    $content .= '</div>' . PHP_EOL;
    echo $content;
  }
}

/**
 * DisplaySearchWorkshopCom()
 * Affiche les groupes de travail répondant à la requête sur $keywords
 * Requête éffectuée sur les discussions du calendrier de groupes de travail ( titre & contenu )
 *
 * @param string $keywords
 * @return string $content
 */
if(!function_exists('DisplaySearchWorkshopCom')) {

  function DisplaySearchWorkshopCom($keywords, $type, $searchtype)
  {
    $data = $GLOBALS['sql_object']->DBSelect(SQL_workshop_com_search(MatchDataFormat($keywords, $searchtype), $type, $searchtype));

    $content = '<div id="MatchWorkshopCom">';
    $content .= '<h4>' . _t('search','workshopcom_module') . '</h4>' . PHP_EOL;
    //if (!is_array($data)) $content .= '<p>' . _t('search','no_result') . '</p>' . PHP_EOL;
    if (!is_array($data)) return true;
    else {
    	$stack = array();
      $content .= '<ul>' . PHP_EOL;
      for($i = 0; $i < count($data); $i++) {
        if ($data[$i]['workcom_parent'] != 0) $id_com = $data[$i]['workcom_parent'];
        else $id_com = $data[$i]['workcom_id'] ;
        if ($type == 'PUBLIC') {
          $link_workcom = HrefMaker(array('rub' => $GLOBALS['links'][U_L]['topic']['linkvalue'], 'id' => $id_com, 'parentid' => $data[$i]['workcom_workshop_id'], 'name' => $data[$i]['workcom_subject']));
          $link_workshop = HrefMaker(array('rub' => $GLOBALS['links'][U_L]['workgroup']['linkvalue'], 'id' => $data[$i]['workcom_workshop_id'], 'name' => $data[$i]['workshop_denomination']));
        }
        if ($type == 'ADMIN') {
          $link_workcom = 'index.php?rub=workshop&amp;todo=det_com&amp;id=' . $data[$i]['workcom_workshop_id'] . '&amp;com_id=' . $id_com;
          $link_workshop = 'index.php?rub=workshop&amp;todo=det&amp;id=' . $data[$i]['workcom_workshop_id'];
        }

        $workcom_title = formatText($data[$i]['workcom_subject'], '2HTML');
        $workshop_title = formatText($data[$i]['workshop_denomination'], '2HTML');
        $pertinence = FormatPertinenceIndice($data[$i]['pertinence']);
        $published_display = _t('statut','published_on') . ' ' . $data[$i]['date_display'];
				
        // prevent duplicates entries
        if(!in_array($data[$i]['workcom_workshop_id'], $stack)) {
	        if ($type == 'ADMIN' || ($type == 'PUBLIC' && $GLOBALS['l21auth']->isWorkgroupUser($data[$i]['workcom_workshop_id'], $GLOBALS['sql_object']))) {
	          $content .= '<li class="matches">' . PHP_EOL;
	          $content .= '<a href="' . $link_workcom . '" title="' . formatText($GLOBALS['links'][U_L]['topic']['desc'] . ' : ' . $workcom_title, '2ATT') . '" >' . $workcom_title . '</a>';
	          $content .= '<span class="infocomp"> - ' . mb_ucfirst($published_display) . '</span>';
	          $content .= '<br />';
	          $content .= '<span class="infoparent">' . mb_ucfirst(_t('name','workshop_module')) . ' : ';
	          $content .= '<a href="' . $link_workshop . '" title="' . formatText($GLOBALS['links'][U_L]['workgroup']['desc'] . ' : ' . $workshop_title, '2ATT') . '" >' . $workshop_title . '</a>';
	          $content .= '</span>';
	          $content .= $pertinence;
	          $content .= '</li>' . PHP_EOL;
	        }
	        array_push($stack, $data[$i]['workcom_workshop_id']); // prevent duplicates entries
        }
      }
      $content .= '</ul>' . PHP_EOL;

      if ($content == '<ul></ul>') return '';
    }
    $content .= '</div>' . PHP_EOL;

    echo $content;
  }
}

/**
 * DisplaySearchSdi()
 * Affiche les indicateurs répondant à la requête sur $keywords
 * Requête éffectuée sur les informations générales des indicateurs
 *
 * @param string $keywords
 * @return string $content
 */
if(!function_exists('DisplaySearchSdi')) {

  function DisplaySearchSdi($keywords, $type, $searchtype)
  {
    $data = $GLOBALS['sql_object']->DBSelect(SQL_sdi_search(MatchDataFormat($keywords, $searchtype), $type, $searchtype));

    if(isset($_REQUEST['rub']) && $_REQUEST['rub']=='dashboard') {
    	$rub = 'dashboard';
    } else {
    	$rub = 'sdi';
    }

    $content = '<div id="MatchSdi">';
    $content .= '<h3><i class="fa fa-paperclip" aria-hidden="true"></i> ' . _t('search','sdi_module') . '</h3>' . PHP_EOL;
    if (!is_array($data)) $content .= '<p>' . _t('search','no_result') . '</p>' . PHP_EOL;
    else {
      $content .= '<ul>' . PHP_EOL;
      for($i = 0; $i < count($data); $i++) {
        if ($type == 'PUBLIC') $link_sdi = HrefMaker(array('rub' => $GLOBALS['links'][U_L]['dashboard']['linkvalue'], 'id' => $data[$i]['sdii_id'], 'parentid' => 1, 'name' => $data[$i]['sdii_name']));
        if ($type == 'ADMIN') $link_sdi = 'index.php?rub='.$rub.'&amp;todo=det&amp;id=' . $data[$i]['sdii_id'];

        $news_title = formatText($data[$i]['sdii_name'], '2HTML');
        $pertinence = FormatPertinenceIndice($data[$i]['pertinence']);
        $published_display = $data[$i]['date_display'];

        $content .= '<li class="matches">' . PHP_EOL;
        $content .= '<a href="' . $link_sdi . '" title="' . formatText($GLOBALS['links'][U_L]['dashboard']['desc'] . ' : ' . $news_title, '2ATT') . '" >' . $news_title . '</a>';
        $content .= '<span class="infocomp"> - ' . mb_ucfirst(_t('statut','published_on')) . ' ' . $published_display . GetStatus($data[$i]['statut']) . '</span>';
        $content .= $pertinence;
        $content .= '</li>' . PHP_EOL;
      }
      $content .= '</ul>' . PHP_EOL;
    }
    $content .= '</div>' . PHP_EOL;
    echo $content;
  }
}

/**
 * DisplaySearchYellowpages()
 * Affiche les organisations de l'annuaire répondant à la requête sur $keywords
 *
 * @param string $keywords
 * @return string $content
 */
if(!function_exists('DisplaySearchYellowpages')) {

  function DisplaySearchYellowpages($keywords, $type, $searchtype)
  {
    $data = $GLOBALS['sql_object']->DBSelect(SQL_yellowpages_search(MatchDataFormat($keywords, $searchtype), $type, $searchtype));

    $content = '<div id="MatchYellowpages">';
    $content .= '<h3><i class="fa fa-list" aria-hidden="true"></i> ' . _t('search','yellowpages_module') . '</h3>' . PHP_EOL;
    if (!is_array($data)) $content .= '<p>' . _t('search','no_result') . '</p>' . PHP_EOL;
    else {
      $content .= '<ul>' . PHP_EOL;
      for($i = 0; $i < count($data); $i++) {
        if ($type == 'PUBLIC') $link_yp = HrefMaker(array('rub' => $GLOBALS['links'][U_L]['directory-detail']['linkvalue'], 'id' => $data[$i]['yellowp_id'], 'name' => $data[$i]['yellowp_name']));
        if ($type == 'ADMIN') $link_yp = 'index.php?rub=yellowpages&amp;todo=det&amp;id=' . $data[$i]['yellowp_id'];

        $yp_title = formatText($data[$i]['yellowp_name'], '2HTML');
        $pertinence = _t('divers','none');
        $published_display = $data[$i]['date_display'];

        $content .= '<li class="matches">' . PHP_EOL;
        $content .= '<a href="' . $link_yp . '" title="' . formatText($GLOBALS['links'][U_L]['directory-detail']['desc'] . ' : ' . $yp_title, '2ATT') . '" >' . $yp_title . '</a>';
        $content .= '<span class="infocomp"> - ' . mb_ucfirst(_t('divers','date_crea')) . ' : ' . $published_display . GetStatus($data[$i]['statut']) . '</span>';
        $content .= '</li>' . PHP_EOL;
      }
      $content .= '</ul>' . PHP_EOL;
    }
    $content .= '</div>' . PHP_EOL;
    echo $content;
  }
}

/**
 * DisplaySearchUser()
 * Affiche les utilisateurs répondant à la requête sur $keywords
 *
 * @param string $keywords
 * @return string $content
 */
if(!function_exists('DisplaySearchUser')) {

  function DisplaySearchUser($keywords, $type, $searchtype)
  {
    $data = $GLOBALS['sql_object']->DBSelect(SQL_user_search(MatchDataFormat($keywords, $searchtype), $type, $searchtype));

    $content = '<div id="MatchUser">';
    $content .= '<h3><i class="fa fa-user" aria-hidden="true"></i> ' . _t('search','user_module') . '</h3>' . PHP_EOL;
    if (!is_array($data)) $content .= '<p>' . _t('search','no_result') . '</p>' . PHP_EOL;
    else {
      $content .= '<ul>' . PHP_EOL;
      for($i = 0; $i < count($data); $i++) {
        if ($type == 'PUBLIC') $link_user = HrefMaker(array('rub' => $GLOBALS['links'][U_L]['user-profile']['linkvalue'], 'id' => $data[$i]['user_id'], 'name' => $data[$i]['user_login']));
        if ($type == 'ADMIN') $link_user = 'index.php?rub=user&amp;todo=det&amp;id=' . $data[$i]['user_id'];

        $user_title = formatText($data[$i]['user_login']);
        if ($type == 'ADMIN') {
        	if(!empty($data[$i]['profile_firstname']) || !empty($data[$i]['profile_lastname'])) $user_title .= ' - ' . formatText($data[$i]['profile_firstname']) . ' ' . formatText($data[$i]['profile_lastname']);
        	$user_title .= ' - ' . formatText($data[$i]['profile_email']);
        }
        $pertinence = _t('divers','none');
        $published_display = $data[$i]['date_display'];

        $content .= '<li class="matches">' . PHP_EOL;
        $content .= '<a href="' . $link_user . '" title="' . formatText($GLOBALS['links'][U_L]['user-profile']['desc'] . ' : ' . $user_title, '2ATT') . '" >' . $user_title . '</a>';
        $content .= '<span class="infocomp"> (' . mb_ucfirst(_t('divers','date_crea')) . ' : ' . $published_display . GetStatus($data[$i]['statut']) . ')</span>';
        $content .= '</li>' . PHP_EOL;
      }
      $content .= '</ul>' . PHP_EOL;
    }
    $content .= '</div>' . PHP_EOL;
    echo $content;
  }
}

/**
 * DisplaySearchTag()
 * Affiche les tags répondant à la requête sur $keywords
 *
 * @param string $keywords
 * @return string $content
 */
if(!function_exists('DisplaySearchTag')) {

	function DisplaySearchTag($keywords, $type, $searchtype)
	{
		$data = $GLOBALS['sql_object']->DBSelect(SQL_tag_search(MatchDataFormat($keywords, $searchtype), $type, $searchtype));

		$content = '<div id="MatchTag">';
		$content .= '<h3><i class="fa fa-tag" aria-hidden="true"></i> ' . _t('search','tag_module') . '</h3>' . PHP_EOL;
		if (!is_array($data)) $content .= '<p>' . _t('search','no_result') . '</p>' . PHP_EOL;
		else {
			$content .= '<ul>' . PHP_EOL;
			for($i = 0; $i < count($data); $i++) {
				if ($type == 'ADMIN') $link = 'index.php?rub=tag&amp;todo=det&amp;id=' . $data[$i]['tag_id'];

				$tag_title = formatText($data[$i]['tag_name']);
				$pertinence = _t('divers','none');
				$published_display = $data[$i]['date_display'];

				$content .= '<li class="matches">' . PHP_EOL;
				$content .= '<a href="' . $link . '">' . $tag_title . '</a>';
				$content .= '<span class="infocomp">(' . mb_ucfirst(_t('divers','date_crea')) . ' : ' . $published_display . ')</span>';
				$content .= '</li>' . PHP_EOL;
			}
			$content .= '</ul>' . PHP_EOL;
		}
		$content .= '</div>' . PHP_EOL;
		echo $content;
	}
}


/**
 * DisplaySearchLevel()
 * Affiche les niveaux répondant à la requête sur $keywords
 *
 * @param string $keywords
 * @return string $content
 */
if(!function_exists('DisplaySearchLevel')) {

  function DisplaySearchLevel($keywords, $type, $searchtype)
  {
    $data = $GLOBALS['sql_object']->DBSelect(SQL_level_search(MatchDataFormat($keywords, $searchtype), $type, $searchtype));

    $content = '<div id="MatchLevel">';
    $content .= '<h3><i class="fa fa-inbox" aria-hidden="true"></i> ' . _t('search','level_module') . '</h3>' . PHP_EOL;
    if (!is_array($data)) $content .= '<p>' . _t('search','no_result') . '</p>' . PHP_EOL;
    else {
      $content .= '<ul>' . PHP_EOL;
      for($i = 0; $i < count($data); $i++) {
        if ($type == 'ADMIN') $link_level = 'index.php?rub=level&amp;todo=det&amp;id=' . $data[$i]['level_id'];

        $level_title = formatText($data[$i]['level_name']);
        $pertinence = _t('divers','none');
        $published_display = $data[$i]['date_display'];

        $content .= '<li class="matches">' . PHP_EOL;
        $content .= '<a href="' . $link_level . '">' . $level_title . '</a>';
        $content .= '<span class="infocomp"> - ' . mb_ucfirst(_t('divers','date_crea')) . ' : ' . $published_display . GetStatus($data[$i]['statut']) . '</span>';
        $content .= '</li>' . PHP_EOL;
      }
      $content .= '</ul>' . PHP_EOL;
    }
    $content .= '</div>' . PHP_EOL;
    echo $content;
  }
}

/**
 * DisplaySearchScale()
 * Affiche les échelles répondant à la requête sur $keywords
 *
 * @param string $keywords
 * @return string $content
 */
if(!function_exists('DisplaySearchScale')) {

  function DisplaySearchScale($keywords, $type, $searchtype)
  {
    $data = $GLOBALS['sql_object']->DBSelect(SQL_scale_search(MatchDataFormat($keywords, $searchtype), $type, $searchtype));

    $content = '<div id="MatchScale">';
    $content .= '<h3><i class="fa fa-bookmark" aria-hidden="true"></i> ' . _t('search','scale_module') . '</h3>' . PHP_EOL;
    if (!is_array($data)) $content .= '<p>' . _t('search','no_result') . '</p>' . PHP_EOL;
    else {
      $content .= '<ul>' . PHP_EOL;
      for($i = 0; $i < count($data); $i++) {
        if ($type == 'ADMIN') $link_scale = 'index.php?rub=scale&amp;todo=det&amp;id=' . $data[$i]['scale_id'];

        $scale_title = formatText($data[$i]['scale_denomination']);
        $pertinence = _t('divers','none');
        $published_display = $data[$i]['date_display'];

        $content .= '<li class="matches">' . PHP_EOL;
        $content .= '<a href="' . $link_scale . '" title="' . formatText($scale_title, '2ATT') . '">' . $scale_title . '</a>';
        $content .= '<span class="infocomp">(' . mb_ucfirst(_t('divers','date_crea')) . ' : ' . $published_display . GetStatus($data[$i]['statut']) . ')</span>';
        $content .= '</li>' . PHP_EOL;
      }
      $content .= '</ul>' . PHP_EOL;
    }
    $content .= '</div>' . PHP_EOL;
    echo $content;
  }
}

/**
 * DisplayQueryTime()
 * Affiche le temps d'execution des requêtes
 *
 * @param float $start_time
 * @param float $end_time
 * @return string $content (echo)
 */
if(!function_exists('DisplayQueryTime')) {

  function DisplayQueryTime($start_time, $end_time)
  {
    $content = '<div id="QueryTime">' . PHP_EOL;
    $content .= _t('search','query_time') . ' ' . (substr($end_time - $start_time, 0, 5)) . ' ' . _t('search','query_seconds');
    $content .= '</div>' . PHP_EOL;
    echo $content;
  }
}

if(!function_exists('DisplayAdvancedSearchForm')) {

  function DisplayAdvancedSearchForm()
  {
  	
  	if(defined('SEARCH_MODE') && stripos(SEARCH_MODE, 'boolean') === false) $disabled=' disabled';
  	else $disabled='';
  	
    $action = $GLOBALS['links'][U_L]['search-result']['linkvalue'];
    $content  = '<form action="index.php" method="get">';
    $content .= '<div id="searchcontent">';
    $content .= '<label for="advsearch" class="labelised">' . _t('search','keywords') . '</label>';
    $content .= '<input type="text" maxlength="200"  class="textfield" name="search" id="advsearch" style="margin-right:30px" />';
    $content .= '<input id="rub_search2" name="rub" type="hidden" value="'.$action.'" />';
    $content .= '<div style="margin:2em 0 2em 5em">';
    $content .= '<input type="radio" class="radio" name="searchtype" id="searchtype_1" value="one" checked="checked "'.$disabled.' /><label class="labelised autowidth" for="searchtype_1">' . _t('search','one_of_them') . '</label>';
    $content .= '<input type="radio" class="radio" name="searchtype" id="searchtype_2" value="all"'.$disabled.' /><label class="labelised autowidth" for="searchtype_2">' . _t('search','all_of_them') . '</label>';
    $content .= '<input type="radio" class="radio" name="searchtype" id="searchtype_3" value="exp"'.$disabled.' /><label class="labelised autowidth" for="searchtype_3">' . _t('search','expression') . '</label>';
    $content .= '<input type="radio" class="radio" name="searchtype" id="searchtype_4" value="expanded"'.$disabled.' /><label class="labelised autowidth" for="searchtype_4">' . _t('search','expanded') . '</label>';
    $content .= '</div>';
    if(!empty($disabled)) { 
    	$content .= '<div class="info">'._t('search', 'query_expansion_enabled').'</div>';
    }
    $content .='<div class="submit-container"><input type="submit" value="'._t('search', 'text_button').'" name="valid" id="validadvsearch" class="submitbut" /></div>'.PHP_EOL;
    $content .= '</div>';
    $content .= '</form>';

    echo $content;
  }
}

if(!function_exists('FormatPertinenceIndice')) {

  function FormatPertinenceIndice($float) {
    $pertinence_i = ($float != 0) ? number_format(round($float, 3), 3, ',', ' ') : _t('search','score_not_good');
    $pertinence= ' <span class="pertinence">- <span class="hint--top hint--medium hint--rounded" aria-label="'.formatText(_t('search','score'), '2ATT').'">' . _t('search','score_abbr') . '</span> : ' . $pertinence_i . '</span>';


    return $pertinence;
  }
}
?>