<?php
/**
 * Workshop module - iCal event
 * 
 * Create iCal event
 * 
 * @package linea21\modules\workshop
 * @author $Author$ - Simon Georget <simon@linea21.com>
 * @version $Id$ 
 * @access public
 * @license http://opensource.org/licenses/gpl-3.0.html
 */

// The script is adapted from https://gist.github.com/jakebellacera/635416
// Copyright goes to his author

$summary = rawurldecode($_GET['summary']);
$description = rawurldecode($_GET['description']);
$date = rawurldecode($_GET['date']);
$uri = rawurldecode($_GET['uri']);
$filename = rawurldecode($_GET['filename']).'.ics';


// Variables used in this script:
//   $summary     - text title of the event
//   $date     - the starting/ending date (in seconds since unix epoch)
//   $uri         - the URL of the event (add http://)
//   $description - text description of the event
//   $filename    - the name of this file for saving (e.g. my-event-name.ics)
//
// Notes:
//  - the UID should be unique to the event, so in this case I'm just using
//    uniqid to create a uid, but you could do whatever you'd like.
//
//  - iCal requires a date format of "yyyymmddThhiissZ". The "T" and "Z"
//    characters are not placeholders, just plain ol' characters. The "T"
//    character acts as a delimeter between the date (yyyymmdd) and the time
//    (hhiiss), and the "Z" states that the date is in UTC time. Note that if
//    you don't want to use UTC time, you must prepend your date-time values
//    with a TZID property. See RFC 5545 section 3.3.5
//
//  - The Content-Disposition: attachment; header tells the browser to save/open
//    the file. The filename param sets the name of the file, so you could set
//    it as "my-event-name.ics" or something similar.
//
//  - Read up on RFC 5545, the iCalendar specification. There is a lot of helpful
//    info in there, such as formatting rules. There are also many more options
//    to set, including alarms, invitees, busy status, etc.
//
//      https://www.ietf.org/rfc/rfc5545.txt

// 1. Set the correct headers for this file
header('Content-type: text/calendar; charset=utf-8');
header('Content-Disposition: attachment; filename=' . $filename);

// 2. Define helper functions

// Converts a unix timestamp to an ics-friendly format
// NOTE: "Z" means that this timestamp is a UTC timestamp. If you need
// to set a locale, remove the "\Z" and modify DTEND, DTSTAMP and DTSTART
// with TZID properties (see RFC 5545 section 3.3.5 for info)
//
// Also note that we are using "g" instead of "H" because iCalendar's Time format
// requires 24-hour time (see RFC 5545 section 3.3.12 for info).
function dateToCal($timestamp) {
  return date('Ymd\Tgis\Z', $timestamp);
}

// Escapes a string of characters
// we escape commas http://tools.ietf.org/html/rfc5545#section-3.3.11
function escapeString($string) {
	$string = preg_replace('/([\,;])/','\\\$1', $string);
	return $string;
}

// 3. Echo out the ics file's contents
?>
BEGIN:VCALENDAR<?php echo "\r\n"; ?>
VERSION:2.0<?php echo "\r\n"; ?>
PRODID:-//hacksw/handcal//NONSGML v1.0//EN<?php echo "\r\n"; ?>
CALSCALE:GREGORIAN<?php echo "\r\n"; ?>
BEGIN:VEVENT<?php echo "\r\n"; ?>
UID:<?php echo uniqid()."\r\n"; ?>
SUMMARY:<?php echo escapeString($summary)."\r\n"; ?>
DESCRIPTION:<?php echo str_replace("\r\n", "\\r\\n", escapeString($description))."\r\n"; ?>
URL;VALUE=URI:<?php echo escapeString($uri)."\r\n"; ?>
DTSTAMP:<?php echo gmdate('Ymd') . 'T' . gmdate('His') . 'Z' . "\r\n"; ?>
DTSTART;VALUE=DATE:<?php echo $date."\r\n"; ?>
DTEND;VALUE=DATE:<?php echo $date."\r\n"; ?>
END:VEVENT<?php echo "\r\n"; ?>
END:VCALENDAR<?php echo "\r\n"; ?>